package it.unimi.dsi.law.util;

import static org.junit.Assert.assertArrayEquals;

/*
 * Copyright (C) 2008-2020 Paolo Boldi, Massimo Santini and Sebastiano Vigna
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import org.junit.Test;

import it.unimi.dsi.fastutil.BigArrays;



//RELEASE-STATUS: DIST

public class NormTest {

	private final double[] a = {  0.0, -1.0/2.0,  0.0, 1.0/2.0 };
	private final double[] b = { -1.0/3.0,  0.0, -1.0/3.0, 0.0 };

	@Test
	public void testZero() {
		assertArrayEquals(new double[3], Norm.L_1.normalize(new double[3], 0), 0);
		assertTrue(BigArrays.equals(BigArrays.wrap(new double[3]), Norm.L_1.normalize(BigArrays.wrap(new double[3]), 0)));
	}

	@Test
	public void testNan() {
		final double[] normalize = Norm.L_1.normalize(new double[3], 1);
		for (final double v : normalize) assertTrue(Double.isNaN(v));

		final double[][] normalizeBig = Norm.L_1.normalize(BigArrays.wrap(new double[3]), 1);
		for (int i = 0; i < 3; i++) assertTrue(Double.isNaN(BigArrays.get(normalizeBig, i)));
	}

	@Test
	public void testL1ComputeABdouble() {
		final double expResult = 1.0 + 2.0 / 3.0;
		final double result = Norm.L_1.compute(a, b);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL1ComputeBigABdouble() {
		final double expResult = 1.0 + 2.0 / 3.0;
		final double result = Norm.L_1.compute(BigArrays.wrap(a), BigArrays.wrap(b));
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL1ComputeAdouble() {
		final double expResult = 1.0;
		final double result = Norm.L_1.compute(a);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL1ComputeBigAdouble() {
		final double expResult = 1.0;
		final double result = Norm.L_1.compute(BigArrays.wrap(a));
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL1Normalize() {
		final double[] v = { 1, 2, -3, 4.5, -5.5 };
		final double expResult = 1.0;
		Norm.L_1.normalize(v, expResult);
		final double result = Norm.L_1.compute(v);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL1NormalizeBig() {
		final double[] v = { 1, 2, -3, 4.5, -5.5 };
		final double expResult = 1.0;
		Norm.L_1.normalize(BigArrays.wrap(v), expResult);
		final double result = Norm.L_1.compute(BigArrays.wrap(v));
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL2ComputeABdouble() {
		final double expResult = Math.sqrt(13.0 / 18.0);
		final double result = Norm.L_2.compute(a, b);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL2ComputeAdouble() {
		final double expResult = Math.sqrt(1.0 / 2.0);
		final double result = Norm.L_2.compute(a);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testL2Normalize() {
		final double[] v = { 1, 2, -3, 4.5, -5.5 };
		final double expResult = 2.0;
		Norm.L_2.normalize(v, expResult);
		final double result = Norm.L_2.compute(v);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testLIComputeABdouble() {
		final double expResult = 1 / 2.;
		final double result = Norm.L_INFINITY.compute(a, b);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testLIComputeAdouble() {
		final double expResult = 1 / 2.;
		final double result = Norm.L_INFINITY.compute(a);
		assertEquals(expResult, result, .0);
	}

	@Test
	public void testLINormalize() {
		final double[] v = { 1, 2, -3, 4.5, -5.5 };
		final double expResult = 2.0;
		Norm.L_INFINITY.normalize(v, expResult);
		final double result = Norm.L_INFINITY.compute(v);
		assertEquals(expResult, result, .0);
	}
}
