package it.unimi.dsi.law.big.stat;

/*
 *  Copyright (C) 2006-2020 Paolo Boldi, Roberto Posenato, Massimo Santini and Sebastiano Vigna
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import static it.unimi.dsi.fastutil.BigArrays.length;
import static it.unimi.dsi.fastutil.BigArrays.wrap;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;

import java.io.File;
import java.io.IOException;

import org.junit.Test;

import it.unimi.dsi.fastutil.BigArrays;
import it.unimi.dsi.fastutil.doubles.DoubleArrays;
import it.unimi.dsi.fastutil.io.BinIO;
import it.unimi.dsi.fastutil.io.TextIO;
import it.unimi.dsi.util.XoRoShiRo128PlusRandom;

//RELEASE-STATUS: DIST

public class KendallTauTest {
	private final static double[][] ordered = wrap(new double[] { 0.0, 1.0, 2.0, 3.0, 4.0 });
	private final static double[][] reverse = wrap(new double[] { 4.0, 3.0, 2.0, 1.0, 0.0 });
	private final static double[][] reverseButOne = wrap(new double[] { 10.0, 9.0, 7.0, 8.0, 6.0 });
	private final static double[][] allOnes = wrap(new double[] { 1.0, 1.0, 1.0 });
	private final static double[][] allZeroes = wrap(new double[] { 0.0, 0.0, 0.0 });

	/** Computes Kendall's τ by brute-force enumeration of all pairs (<var>i</var>, <var>j</var>), <var>i</var>&lt;<var>j</var>.
	 *
	 * @param v0 the first score vector.
	 * @param v1 the second score vector.
	 * @return Kendall's τ.
	 */
	public static double compute(final double[][] bv0, final double[][] bv1) {
		final double[] v0 = new double[(int) length(bv0)], v1 = new double[(int) length(bv1)];
		BigArrays.copyFromBig(bv0, 0, v0, 0, v0.length);
		BigArrays.copyFromBig(bv1, 0, v1, 0, v1.length);
		return it.unimi.dsi.law.stat.KendallTauTest.compute(v0, v1);
	}

	@Test
	public void testComputeOrdered() {
		final double expResult = compute(ordered, ordered); // (10.0 - 0.0) / 10.0;
		final double result = KendallTau.INSTANCE.compute(ordered, ordered);
		assertEquals(expResult, result, 0.0);
	}

	@Test
	public void testComputeWithReverse() {
		final double expResult = compute(ordered, reverse);//(0 - 10.0) / 10.0;
		final double result = KendallTau.INSTANCE.compute(ordered, reverse);
		assertEquals(expResult, result, 0.0);
	}

	@Test
	public void testComputeWithReverseButOne() {
		final double expResult = compute(ordered, reverseButOne);//(1.0 - 9.0) / 10.0;
		final double result = KendallTau.INSTANCE.compute(ordered, reverseButOne);
		assertEquals(expResult, result, 0.0);
	}

	@Test
	public void testRandom() {
		final XoRoShiRo128PlusRandom XoRoShiRo128PlusRandom = new XoRoShiRo128PlusRandom(0);
		final double v0[] = new double[1000];
		final double v1[] = new double[1000];
		for(int i = v0.length; i-- != 0;) {
			v0[i] = XoRoShiRo128PlusRandom.nextDouble();
			v1[i] = XoRoShiRo128PlusRandom.nextDouble();
		}
		final double expResult = compute(wrap(v0), wrap(v1));
		final double result = KendallTau.INSTANCE.compute(wrap(v0), wrap(v1));
		assertEquals(expResult, result, 1E-9);
		DoubleArrays.reverse(v0);
		DoubleArrays.reverse(v1);
		assertEquals(expResult, KendallTau.INSTANCE.compute(wrap(v0), wrap(v1)), 1E-9);
	}

	@Test
	public void testRandomWithTies() {
		final XoRoShiRo128PlusRandom XoRoShiRo128PlusRandom = new XoRoShiRo128PlusRandom(0);
		final double v0[] = new double[1000];
		final double v1[] = new double[1000];
		for(int i = v0.length; i-- != 0;) {
			v0[i] = XoRoShiRo128PlusRandom.nextInt(10);
			v1[i] = XoRoShiRo128PlusRandom.nextInt(10);
		}
		final double expResult = compute(wrap(v0), wrap(v1));
		final double result = KendallTau.INSTANCE.compute(wrap(v0), wrap(v1));
		assertEquals(expResult, result, 1E-9);
		DoubleArrays.reverse(v0);
		DoubleArrays.reverse(v1);
		assertEquals(expResult, KendallTau.INSTANCE.compute(wrap(v0), wrap(v1)), 1E-9);
	}


	@Test
	public void testAllTies() {
		assertEquals(1.0, KendallTau.INSTANCE.compute(allOnes, allZeroes), 0.0);
	}

	@Test
	public void testOneAllTies() {
		assertTrue(Double.isNaN(KendallTau.INSTANCE.compute(allOnes, wrap(new double[] { 0.0, 1.0, 2.0 }))));
	}

	@Test
	public void testInputType() throws IOException {
		final File a = File.createTempFile(KendallTauTest.class.getSimpleName(), "a");
		a.deleteOnExit();
		final File b = File.createTempFile(KendallTauTest.class.getSimpleName(), "b");
		b.deleteOnExit();
		BinIO.storeInts(new int[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeInts(new int[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.computeInts(a.toString(), b.toString()), 0);
		assertEquals(-1, KendallTau.INSTANCE.computeInts(a.toString(), b.toString(), true), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Integer.class, b.toString(), Integer.class), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Integer.class, b.toString(), Integer.class, true), 0);
		BinIO.storeInts(new int[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.computeInts(a.toString(), b.toString()), 0);
		assertEquals(1, KendallTau.INSTANCE.computeInts(a.toString(), b.toString(), true), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Integer.class, b.toString(), Integer.class), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Integer.class, b.toString(), Integer.class, true), 0);

		BinIO.storeLongs(new long[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeLongs(new long[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.computeLongs(a.toString(), b.toString()), 0);
		assertEquals(-1, KendallTau.INSTANCE.computeLongs(a.toString(), b.toString(), true), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Long.class, b.toString(), Long.class), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Long.class, b.toString(), Long.class, true), 0);
		BinIO.storeLongs(new long[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.computeLongs(a.toString(), b.toString()), 0);
		assertEquals(1, KendallTau.INSTANCE.computeLongs(a.toString(), b.toString(), true), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Long.class, b.toString(), Long.class), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Long.class, b.toString(), Long.class, true), 0);

		BinIO.storeFloats(new float[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeFloats(new float[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.computeFloats(a.toString(), b.toString()), 0);
		assertEquals(-1, KendallTau.INSTANCE.computeFloats(a.toString(), b.toString(), true), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Float.class, b.toString(), Float.class), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Float.class, b.toString(), Float.class, true), 0);
		BinIO.storeFloats(new float[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.computeFloats(a.toString(), b.toString()), 0);
		assertEquals(1, KendallTau.INSTANCE.computeFloats(a.toString(), b.toString(), true), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Float.class, b.toString(), Float.class), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Float.class, b.toString(), Float.class, true), 0);

		BinIO.storeDoubles(new double[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeDoubles(new double[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.computeDoubles(a.toString(), b.toString()), 0);
		assertEquals(-1, KendallTau.INSTANCE.computeDoubles(a.toString(), b.toString(), true), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Double.class, b.toString(), Double.class), 0);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Double.class, b.toString(), Double.class, true), 0);
		BinIO.storeDoubles(new double[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.computeDoubles(a.toString(), b.toString(), true), 0);
		assertEquals(1, KendallTau.INSTANCE.computeDoubles(a.toString(), b.toString()), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Double.class, b.toString(), Double.class), 0);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Double.class, b.toString(), Double.class, true), 0);

		BinIO.storeInts(new int[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeDoubles(new double[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Integer.class, b.toString(), Double.class), 0);
		BinIO.storeDoubles(new double[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Integer.class, b.toString(), Double.class), 0);

		BinIO.storeDoubles(new double[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeLongs(new long[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), Double.class, b.toString(), Long.class), 0);
		BinIO.storeLongs(new long[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), Double.class, b.toString(), Long.class), 0);

		TextIO.storeDoubles(new double[] { 0, 1, 2, 3, 4 }, a);
		BinIO.storeLongs(new long[] { 4, 3, 2, 1, 0 }, b);
		assertEquals(-1, KendallTau.INSTANCE.compute(a.toString(), String.class, b.toString(), Long.class), 0);
		BinIO.storeLongs(new long[] { 0, 1, 2, 3, 4 }, b);
		assertEquals(1, KendallTau.INSTANCE.compute(a.toString(), String.class, b.toString(), Long.class), 0);

		a.delete();
		b.delete();
	}
}
