package it.unimi.dsi.law.stat;

/*
 * Copyright (C) 2010-2020 Paolo Boldi, Massimo Santini and Sebastiano Vigna
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import java.io.IOException;

import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.SimpleJSAP;
import com.martiansoftware.jsap.UnflaggedOption;

import it.unimi.dsi.webgraph.ImmutableGraph;
import it.unimi.dsi.webgraph.LazyIntIterator;
import it.unimi.dsi.webgraph.NodeIterator;

// RELEASE-STATUS: DIST

/** Computes Pearson assortativities between the list of degrees of sources and targets of
 * arcs of a graph. */

public class PearsonAssortativity {

	protected PearsonAssortativity() {}

	/** Returns the Pearson assortativities of the specified graph.
	 *
	 * @param graph
	 * @return Pearson assortativities of <code>graph</code> in the following order: (+/+, -/+, -/-, +/-, -+/-+).
	 */
	public static double[] compute(final ImmutableGraph graph) {
		final int[] indegree = new int[graph.numNodes()];
		final int[] outdegree = new int[graph.numNodes()];
		final int[] degree = new int[graph.numNodes()];
		final double[] result = new double[5];

		// Compute indegrees and outdegrees.

		final NodeIterator nodeIterator = graph.nodeIterator();
		for(int i = graph.numNodes(); i-- != 0 ;) {
			final int x = nodeIterator.nextInt();
			outdegree[x] = nodeIterator.outdegree();
			final LazyIntIterator successors = nodeIterator.successors();
			int s;
			while((s = successors.nextInt()) != -1) indegree[s]++;
		}

		// Compute assortativities

		result[0] = pearsonIndex(outdegree, outdegree, graph, indegree, outdegree);
		result[1] = pearsonIndex(indegree, outdegree, graph, indegree, outdegree);
		result[2] = pearsonIndex(indegree, indegree, graph, indegree, outdegree);
		result[3] = pearsonIndex(outdegree, indegree, graph, indegree, outdegree);

		for (int i = graph.numNodes() - 1; i >= 0; i--) degree[i] = indegree[i] + outdegree[i];
		result[4] = pearsonIndex(degree, degree, graph, indegree, outdegree);

		return result;
	}

	/** Compute Pearson's correlation index using Definition 3.1 of <i>Degree-degree dependencies
	 *  in directed networks with heavy-tailed degrees</i> by Pim van der Hoorn and Nelly Litvak (Internet Mathematics,
	 *  11:115-179, 2015).
	 *
	 * @param dAlpha
	 * @param dBeta
	 * @return
	 */
	private static double pearsonIndex(final int[] dAlpha, final int[] dBeta, final ImmutableGraph graph, final int[] indegree, final int[] outdegree) {
		double pa2, pa, mb2, mb, ab;
		pa2 = pa = mb2 = mb = ab = 0;
		final long numArcs = graph.numArcs();
		final NodeIterator nodeIterator = graph.nodeIterator();
		for (int i = graph.numNodes(); i-- != 0;) {
			final int source = nodeIterator.nextInt();
			double t = ((double)outdegree[source]) * dAlpha[source];
			pa += t;
			pa2 += t * dAlpha[source];
			t = ((double)indegree[source]) * dBeta[source];
			mb += t;
			mb2 += t * dBeta[source];
			final LazyIntIterator successors = nodeIterator.successors();
			int target;
			while ((target = successors.nextInt()) != -1)
				ab += ((double)dAlpha[source]) * dBeta[target];
		}
		final double sa = Math.sqrt(pa2 / numArcs - (pa / numArcs) * (pa / numArcs));
		final double sb = Math.sqrt(mb2 / numArcs - (mb / numArcs) * (mb / numArcs));
		final double rhat = (pa / (sa * numArcs)) * (mb / (sb * numArcs));
		return (ab / numArcs) / (sa * sb) - rhat;

	}

	public static double[] computeAlt(final ImmutableGraph graph) {
		final int[] indegree = new int[graph.numNodes()];
		final int[] outdegree = new int[graph.numNodes()];
		final int[] degree = new int[graph.numNodes()];
		final double[] result = new double[5];

		// Compute indegrees and outdegrees.

		final NodeIterator nodeIterator = graph.nodeIterator();
		for(int i = graph.numNodes(); i-- != 0 ;) {
			final int x = nodeIterator.nextInt();
			outdegree[x] = nodeIterator.outdegree();
			final LazyIntIterator successors = nodeIterator.successors();
			int s;
			while((s = successors.nextInt()) != -1) indegree[s]++;
		}

		// Compute assortativities

		result[0] = pearsonIndexAlt(outdegree, outdegree, graph);
		result[1] = pearsonIndexAlt(indegree, outdegree, graph);
		result[2] = pearsonIndexAlt(indegree, indegree, graph);
		result[3] = pearsonIndexAlt(outdegree, indegree, graph);

		for (int i = graph.numNodes() - 1; i >= 0; i--) degree[i] = indegree[i] + outdegree[i];
		result[4] = pearsonIndexAlt(degree, degree, graph);

		return result;
	}

	private static double pearsonIndexAlt(final int[] dAlpha, final int[] dBeta, final ImmutableGraph graph) {
		double suma, suma2, sumb, sumb2, sumab;
		suma = suma2 = sumb = sumb2 = sumab = 0;
		final long numArcs = graph.numArcs();
		final NodeIterator nodeIterator = graph.nodeIterator();
		for (int i = graph.numNodes(); i-- != 0;) {
			final int source = nodeIterator.nextInt();
			int target;
			final LazyIntIterator successors = nodeIterator.successors();
			while ((target = successors.nextInt()) != -1) {
				suma += dAlpha[source];
				suma2 += ((double)dAlpha[source]) * dAlpha[source];
				sumb += dBeta[target];
				sumb2 += ((double)dBeta[target]) * dBeta[target];
				sumab += ((double)dAlpha[source]) * dBeta[target];
			}
		}
		final double sa = Math.sqrt(suma2 / numArcs - (suma / numArcs) * (suma / numArcs));
		final double sb = Math.sqrt(sumb2 / numArcs - (sumb / numArcs) * (sumb / numArcs));
		return (sumab / numArcs - (suma / numArcs) * (sumb / numArcs)) / (sa * sb);
	}

	public static void main(final String[] arg) throws NumberFormatException, IOException, JSAPException {
		final SimpleJSAP jsap = new SimpleJSAP(PearsonAssortativity.class.getName(),
			"Prints Pearson's assortativities of a graph.",
			new Parameter[] {
				new UnflaggedOption("basename", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.REQUIRED, JSAP.NOT_GREEDY, "The basename of a graph."),
			}
		);

		final JSAPResult jsapResult = jsap.parse(arg);
		if (jsap.messagePrinted()) System.exit(1);

		final ImmutableGraph graph = ImmutableGraph.loadOffline(jsapResult.getString("basename"));
		final double[] result = compute(graph);
		System.out.println("+/+: " + result[0]);
		System.out.println("-/+: " + result[1]);
		System.out.println("-/-: " + result[2]);
		System.out.println("+/-: " + result[3]);
		System.out.println("-+/-+: " + result[4]);
	}
}
