package it.unimi.dsi.law.fibrations;

/*
 * Copyright (C) 2005-2020 Sebastiano Vigna
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntArrays;
import it.unimi.dsi.fastutil.ints.IntComparator;
import it.unimi.dsi.fastutil.io.BinIO;
import it.unimi.dsi.webgraph.ImmutableGraph;
import it.unimi.dsi.webgraph.NodeIterator;

// RELEASE-STATUS: DIST

/** Static methods to compute the minimum opfibration base of a given graph. */

public class PreProcessedMinimumBase {
	private final static Logger LOGGER = LoggerFactory.getLogger(PreProcessedMinimumBase.class);
	private final static boolean ASSERTS = false;

	/** The bit used to mark alternately blocks. */
	private final static int BLOCK_MARKER = 1 << 31;
	/** The mask containing all bits not used to mark. */
	private final static int LABEL_MASK = ~BLOCK_MARKER;

	/** A comparator that orders lexicographically by indegree, then by node colour, and finally by multiset
	 * of outgoing arc colours (assuming that successors are sorted by colour. It is used in the
	 * preprocessing phase.
	 */

	private static final class OutdegreeNodeColourArcColourComparator implements IntComparator {
		private final int[][] succ;
		private final NodeColouringStrategy nodeColouring;
		private final ArcColouringStrategy arcColouring;

		private OutdegreeNodeColourArcColourComparator(final int[][] succ, final NodeColouringStrategy nodeColouring, final ArcColouringStrategy arcColouring) {
			this.arcColouring = arcColouring;
			this.succ = succ;
			this.nodeColouring = nodeColouring;
		}

		@Override
		public int compare(final int x, final int y) {
			int t;
			// First order by indegree...
			if ((t = succ[x].length - succ[y].length) != 0) return t;
			// ...then by node colour...
			if (nodeColouring != null && (t = nodeColouring.colour(x) - nodeColouring.colour(y)) != 0) return t;
			// ...and finally by multiset of outgoing colours (successors are assumed to be sorted by colour).
			if (arcColouring != null) {
				final int[] succx = succ[x], succy = succ[y];
				final int l = succx.length;
				for(int i = 0; i < l; i++)
					if ((t = arcColouring.colour(x, succx[i]) - arcColouring.colour(y, succy[i])) != 0) return t;
			}
			return 0;
		}
	}

	/** A comparator that orders successors of a given source by the colour of the arc leading to them. Note that since there is no
	 * explicit notion of arc in WebGraph, we resort to an attribute {@link #source} that must be
	 * set before using the comparator. Then, the integers passed to {@link #compare(int, int)} must be
	 * successors of {@link #source}, and the colour of the respective arc will be used to order them.
	 */
	private static final class ArcColourComparator implements IntComparator {
		public int source;
		private final ArcColouringStrategy arcColouring;

		public ArcColourComparator(final ArcColouringStrategy arcColouring) {
			this.arcColouring = arcColouring;
		}

		@Override
		public int compare(final int x, final int y) {
			return arcColouring.colour(source, x) - arcColouring.colour(source, y);
		}
	}

	/** A comparator that orders nodes by their label. The {@link PreProcessedMinimumBase#BLOCK_MARKER} is stripped.
	 * The array of labels is settable using the attribute {@link #label}. */
	private static final class LabelComparator implements IntComparator {
		public int[] label;

		@Override
		public int compare(final int x, final int y) {
			return label[x & LABEL_MASK] - label[y & LABEL_MASK];
		}
	}

	/** A comparator that orders list of successors by comparing lexicographically their list of labels. The lists
	 * are assumed to have the same length (an assertion will fail otherwise). The array of labels is
	 * settable using the attribute {@link #label}. */
	private static final class LabelListComparator implements IntComparator {
		public int[] label;
		private final int[][] succ;

		public LabelListComparator(final int[][] succ) {
			this.succ = succ;
		}

		@Override
		public int compare(final int x, final int y) {
			final int[] succx = succ[x], succy = succ[y];
			final int lx = succx.length;
			int t;

			if (ASSERTS) assert lx == succ[y].length;

			for(int i = 0; i < lx; i++) if ((t = label[succx[i] & LABEL_MASK] - label[succy[i] & LABEL_MASK]) != 0) return t;
			return 0;
		}
	}

	/** Returns a labelling of an immutable graph such that two nodes have the same label iff they
	 * have the same universal opfibration. Note that the labelling is surjective&mdash;if a node
	 * has label <var>k</var>, there are nodes with label <var>j</var>, for every 0&le;<var>j</var>&le;<var>k</var>.
	 *
	 * @param g an immutable graph.
	 * @param nodeColouring a colouring for the nodes, or {@code null}.
	 * @param arcColouring a colouring for the arcs, or {@code null}.
	 * @return an array of integers labelling the graph so that two nodes have the same label iff they
	 * have the same universal opfibration.
	 */

	public static int[] universalFibrationLabelling(final ImmutableGraph g, final NodeColouringStrategy nodeColouring, final ArcColouringStrategy arcColouring) {
		final int n = g.numNodes();
		if (n == 0) return IntArrays.EMPTY_ARRAY;
		final int[][] label = new int[2][n];
		final int[] perm = new int[n];
		int[] curLabel = label[0], newLabel = label[1];

		// Preallocate and fill successor lists, ordering them by arc colours.
		final int succ[][] = new int[n][];
		final NodeIterator nodeIterator = g.nodeIterator();
		int d;
		final ArcColourComparator arcColourComparator = new ArcColourComparator(arcColouring);

		for(int i = 0; i < n; i++) {
			nodeIterator.nextInt();
			d = nodeIterator.outdegree();
			succ[i] = d != 0 ? new int[d] : IntArrays.EMPTY_ARRAY;
			System.arraycopy(nodeIterator.successorArray(), 0, succ[i], 0, d);

			if (arcColouring != null) {
				// We sort each list of successors using the arc colour, so to get canonical representatives for the multisets of colours.
				arcColourComparator.source = i;
				IntArrays.quickSort(succ[i], 0, d, arcColourComparator);
			}
		}

		/* Preprocessing for first-round label assignment; we order lexicographically
		 * by outdegree, node colour and multiset of outgoing colours. */
		for(int i = n; i-- != 0;) perm[i] = i;
		final OutdegreeNodeColourArcColourComparator outdegreeNodeColourArcColourComparator = new OutdegreeNodeColourArcColourComparator(succ, nodeColouring, arcColouring);
		IntArrays.quickSort(perm, 0, n, outdegreeNodeColourArcColourComparator);

		int maxLabel;
		curLabel[perm[0]] = maxLabel = 0;
		/* Now we initialise labels so that only nodes with the same node colour
		 * and multiset of incoming arc colours get the same label. */
		for(int i = 1; i < n; i++)
			curLabel[perm[i]] = outdegreeNodeColourArcColourComparator.compare(perm[i - 1], perm[i]) == 0 ? maxLabel : ++maxLabel;

		/* Since nodes with the same label have the same multiset of outgoing colours, we
		 * can forget the colours completely: we need just to remember which consecutive
		 * blocks of successors share the same colour. We use the sign bit as an alternating
		 * marker to this purpose, starting from the end of the list. */
		if (arcColouring != null) {
			int[] s;
			for(int i = n; i-- != 0;) {
				s = succ[i];
				d = s.length;
				if (d < 2) continue;
				int arcColour = arcColouring.colour(i, s[--d]);
				int marker = 0;
				while(d-- != 0) {
					if (arcColouring.colour(i, s[d]) != arcColour) {
						marker = marker == 0 ? BLOCK_MARKER : 0;
						arcColour = arcColouring.colour(i, s[d]);
					}
					s[d] |= marker;
				}
			}
		}

		LOGGER.info("After preprocessing: " + (maxLabel + 1) + " labels");

		final LabelComparator labelComparator = new LabelComparator();
		final LabelListComparator labelListComparator = new LabelListComparator(succ);
		// We iteratively relabel nodes. Note that preprocessing is equivalent to a first iteration.
		for(int k = 0; k < n - 2; k++) {
			LOGGER.info("Starting phase " + k + "...");
			curLabel = labelComparator.label = labelListComparator.label = label[k % 2];
			newLabel = label[1 - k % 2];
			int start, end = 0, startLabel, newMaxLabel = -1;

			for(int l = 0; l <= maxLabel; l++) {
				// We treat separately classes of nodes with the same label.
				startLabel = curLabel[perm[start = end++]];
				// Find delimiter of the current block
				while(end < n && curLabel[perm[end]] == startLabel) end++;

				// Blocks with one element cannot be split anymore, so we don't do anything else.
				if (end - start > 1) {
					// Sort each list in the class, given that their length is greater than 1.
					if (succ[perm[start]].length > 1)
						for(int i = start; i < end; i++) {
							boolean positive;
							int t;
							final int[] s = succ[perm[i]];
							for(int j = s.length - 1; j > 0;) {
								positive = s[t = j] >= 0;
								do j--; while(j >= 0 && positive == (s[j] >= 0));

								if (t - j > 1) IntArrays.quickSort(s, j + 1, t + 1, labelComparator);
							}
						}

					// Sort the class
					IntArrays.quickSort(perm, start, end, labelListComparator);
				}

				// Relabel
				newLabel[perm[start]] = ++newMaxLabel;
				for(int i = start + 1; i < end; i++)
					newLabel[perm[i]] = labelListComparator.compare(perm[i - 1], perm[i]) == 0 ? newMaxLabel : ++newMaxLabel;

			}
			//BinIO.storeInts(newLabel, g.basename() + ".E_" + k);
			if (maxLabel == newMaxLabel) break;
			maxLabel = newMaxLabel;
		}

		return newLabel;
	}

	public static void main(final String arg[]) throws IOException {
		if (arg.length == 1) System.out.println(IntArrayList.wrap(universalFibrationLabelling(ImmutableGraph.load(arg[0]), null, null)));
		else if (arg.length == 2) BinIO.storeInts(universalFibrationLabelling(ImmutableGraph.load(arg[0]), null, null), arg[1]);
		else if (arg.length == 3) {
			final int[] d = BinIO.loadInts(arg[1]);
			BinIO.storeInts(universalFibrationLabelling(ImmutableGraph.load(arg[0]), null, (x, y) -> d[y]), arg[2]);
		}
		else throw new IllegalArgumentException();
	}
}
