package it.unimi.dsi.law.big.util;

/*
 * Copyright (C) 2004-2020 Paolo Boldi, Massimo Santini and Sebastiano Vigna
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import static it.unimi.dsi.fastutil.BigArrays.copy;
import static it.unimi.dsi.fastutil.BigArrays.get;
import static it.unimi.dsi.fastutil.BigArrays.length;
import static it.unimi.dsi.fastutil.BigArrays.set;

import it.unimi.dsi.fastutil.longs.LongBigArrays;

// RELEASE-STATUS: DIST

/** Computes the number of discordances between two integer score big vectors
 * using Knight's O(<var>n</var>&nbsp;log&nbsp;<var>n</var>)
 * MergeSort-based algorithm.
 *
 * <p>This class is just a copy of {@link ExchangeCounter} using integer big vectors.
 *
 * @see ExchangeCounter
 */
public class IntegerExchangeCounter {
	/** Below this number of elements we use insertion sort. */
	private final static int SMALL = 32;
	/** A support big array used by MergeSort. Must be at least as large as {@link #perm}. */
	private final long[][] temp;
	/** The score big vector used to perform comparisons. */
	private final int[][] v;
	/** A big array of longs (representing a previously built order). */
	private final long perm[][];

	/** Creates a new exchange counter with a provided support big array.
	 *
	 * <P>This constructor avoids the need to allocate a support big array, in case one is already available.
	 *
	 * @param perm the big array to be sorted.
	 * @param v the score integer big vector used to compare the element of {@code perm}.
	 * @param support a big array that will be used as temporary storage during the computation (its content will be erased);
	 * must be at least as long as <code>a</code>.
	 */
	public IntegerExchangeCounter(final long perm[][], final int[][] v, final long[][] support) {
		this.perm = perm;
		this.v = v;
		if (length(support) < length(perm)) throw new IllegalArgumentException("The support array length (" + length(support) + ") is smaller than the main array length (" + length(perm) + ")");
		this.temp = support;
	}

	/** Creates a new exchange counter.
	 *
	 * @param perm the big array to be sorted.
	 * @param v the score integer big vector used to compare the element of {@code perm}.
	 */
	public IntegerExchangeCounter(final long perm[][], final int[][] v) {
		this(perm, v, LongBigArrays.newBigArray(length(perm)));
	}

	/** Computes the number of exchanges.
	 *
	 * <P>Note that a call to this method will actually sort the permutation at creation time.
	 *
	 * @return the number of exchanges that will order the permutation provided at creation time using the score vector
	 * provided at creation time.
	 */
	public long count() {
		return count(0, length(perm));
	}

	/** Orders a subarray of {@link #perm} and returns the number of exchanges.
	 *
	 * @param offset the starting element to order.
	 * @param length the number of elements to order.
	 * @return the number of exchanges in the subarray.
	 */


	private long count(final long offset, final long length) {
		long exchanges = 0;
		final long[][] perm = this.perm;

		if (length < SMALL) {
			final long end = offset + length;
			for (long i = offset; ++i < end;) {
				final long t = get(perm, i);
				long j = i;
				for (long u = get(perm, j - 1); get(v, t) < get(v, u); u = get(perm, j - 1)) {
					exchanges++;
					set(perm, j--, u);
					if (offset == j) break;
				}
				set(perm, j, t);
			}

			return exchanges;
		}

		final long length0 = length / 2, length1 = length - length / 2, middle = offset + length0;
		exchanges += count(offset, length0);
		exchanges += count(middle, length1);

		/* If the last element of the first subarray is smaller than the first element of
		 * the second subarray, there is nothing to do.  */
		if (get(v, get(perm, middle - 1)) >= get(v, get(perm, middle))) {
			// We merge the lists into temp, adding the number of forward moves to exchanges.
			long i = 0, j = 0, k = 0;
			while(j < length0 && k < length1) {
				if (get(v, get(perm, offset + j)) <= get(v, get(perm, middle + k))) {
					set(temp, i, get(perm, offset + j++));
				}
				else {
					set(temp, i, get(perm, middle + k++));
					exchanges += length0 - j;
				}
				i++;
			}

			copy(perm, offset + j, perm, offset + i, length0 - j);
			copy(temp, 0, perm, offset, i);
		}
		return exchanges;
	}
}
