package it.unimi.dsi.law.big.graph;

/*
 * Copyright (C) 2010-2020 Paolo Boldi, Massimo Santini and Sebastiano Vigna
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import static it.unimi.dsi.fastutil.BigArrays.decrementAndGet;
import static it.unimi.dsi.fastutil.BigArrays.get;
import static it.unimi.dsi.fastutil.BigArrays.grow;
import static it.unimi.dsi.fastutil.BigArrays.incrementAndGet;
import static it.unimi.dsi.fastutil.BigArrays.length;
import static it.unimi.dsi.fastutil.BigArrays.set;
import static it.unimi.dsi.fastutil.BigArrays.swap;

import java.io.DataOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.lang.Thread.UncaughtExceptionHandler;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicLongArray;

import org.apache.commons.lang.mutable.MutableDouble;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.SimpleJSAP;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;

import it.unimi.dsi.Util;
import it.unimi.dsi.big.webgraph.ImmutableGraph;
import it.unimi.dsi.big.webgraph.LazyLongIterator;
import it.unimi.dsi.big.webgraph.NodeIterator;
import it.unimi.dsi.big.webgraph.algo.EliasFanoCumulativeOutdegreeList;
import it.unimi.dsi.bits.Fast;
import it.unimi.dsi.fastutil.BigArrays;
import it.unimi.dsi.fastutil.booleans.BooleanBigArrays;
import it.unimi.dsi.fastutil.doubles.DoubleArrayList;
import it.unimi.dsi.fastutil.ints.IntArrays;
import it.unimi.dsi.fastutil.io.BinIO;
import it.unimi.dsi.fastutil.io.FastBufferedOutputStream;
import it.unimi.dsi.fastutil.longs.AbstractLong2IntMap;
import it.unimi.dsi.fastutil.longs.Long2IntMap;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongArrays;
import it.unimi.dsi.fastutil.longs.LongBigArrays;
import it.unimi.dsi.fastutil.objects.ObjectIterator;
import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.util.XoRoShiRo128PlusRandom;

// RELEASE-STATUS: DIST

/**
 * A big implementation of the <em>layered label propagation</em> algorithm described by by Paolo
 * Boldi, Sebastiano Vigna, Marco Rosa, Massimo Santini, and Sebastiano Vigna in &ldquo;Layered
 * label propagation: A multiresolution coordinate-free ordering for compressing social
 * networks&rdquo;, <i>Proceedings of the 20th international conference on World Wide Web</i>, pages
 * 587&minus;596, ACM, 2011.
 *
 * <p>
 * This implementation uses {@linkplain BigArrays big arrays} to provide support for
 * {@linkplain ImmutableGraph big graphs} (i.e., with more than 2<sup>31</sup> nodes). For the rest,
 * it is functionally identical to {@link it.unimi.dsi.law.graph.LayeredLabelPropagation}. However,
 * it supports outdegrees smaller than 2<sup>31</sup>, only.
 *
 * <h2>Memory requirements</h2>
 *
 * <p>
 * This class requires 25 bytes per node (three longs and a boolean), plus the memory that is
 * necessary to load the graph, which however can be just
 * {@link ImmutableGraph#loadMapped(CharSequence, ProgressLogger) memory-mapped}.
 *
 * @author Paolo Boldi
 * @author Marco Rosa
 * @author Massimo Santini
 * @author Sebastiano Vigna
 * @see it.unimi.dsi.law.graph.LayeredLabelPropagation
 */

public class LayeredLabelPropagation {

	private final static Logger LOGGER = LoggerFactory.getLogger(LayeredLabelPropagation.class);

	/** The list of default &gamma; values. It must be kept in sync with the {@link #main(String[])} default parameters. */
	public static final double[] DEFAULT_GAMMAS = { 1., 1./2, 1./4, 1./8, 1./16, 1./32, 1./64, 1./128, 1./256, 1./512, 1./1024, 0 };

	/** The format used to print &gamma;'s. */
	private static final DecimalFormat GAMMA_FORMAT = new java.text.DecimalFormat("0.############");

	/** The default maximum number of updates. */
	public static final int MAX_UPDATES = 100;

	/** The minimum gain in the Hamiltonian. Under this threshold we stop. */
	private final static double GAIN_TRESHOLD = 0.001;

	/** The update list will be shuffled by blocks of this size, to ensure some locality. */
	private static final int SHUFFLE_GRANULARITY = 100000;

	/** A symmetric, loopless graph. */
	private final ImmutableGraph symGraph;

	/** The number of nodes of {@link #symGraph}. */
	private final long n;

	/** The label of each node. After a call to {@link #computePermutation(int, double[], String)}
	 * this field contains the final list of labels. */
	private AtomicLongArray[] label;

	/** Volume of each current cluster, indexed by label (many will be zeroes). */
	private final AtomicLongArray[] volume;

	/** The chosen update order. */
	private final long[][] updateList;

	/** The objective function (Hamiltonian of the potts model). */
	private final double[] objectiveFunction;

	/** The objective function (Hamiltonian of the potts model). */
	private final MutableDouble gapCost;

	/** The random-number generator. */
	private final XoRoShiRo128PlusRandom r;

	/** The basename of temporary files containing labellings for various &gamma;'s. */
	private File labelling;

	/** If true, the user has set a basename for label files, and such files must not be deleted. */
	private boolean labelBasenameSet;

	/** A virtual permutation applied to the graph, or {@code null} for no permutation. */
	private final long[][] startPerm;

	/** Whether to perform an exactly reproducible run in case {@link #startPerm} is not {@code null} (slower). */
	private final boolean exact;

	/** The number of threads used in the computation. */
	private final int numberOfThreads;

	/** The random seed. */
	private final long seed;

	/** For each note, true iff at least one of the successors changed its label. */
	private final boolean[][] canChange;

	/** The number of nodes that changed their label in the current iteration. */
	private final AtomicLong modified;

	/** A simple exception handler that stores the thrown exception in {@link #threadException}. */
	private final SimpleUncaughtExceptionHandler simpleUncaughtExceptionHandler;

	/** One of the throwables thrown by some of the threads, if at least one thread has thrown a throwable. */
	private volatile Throwable threadException;

	/** The current update. */
	private int update;

	/** The starting node of the next chunk of nodes to be processed. */
	protected long nextNode;
	/** The number of arcs before {@link #nextNode}. */
	protected long nextArcs;
	/** The outdegrees cumulative function. */
	protected final EliasFanoCumulativeOutdegreeList cumulativeOutdegrees;


	/** Creates a new instance.
	 *
	 * @param symGraph a symmetric, loopless graph.
	 * @param seed a random seed.
	 */
	public LayeredLabelPropagation(final ImmutableGraph symGraph, final long seed) throws IOException {
		this(symGraph, null, seed, false);
	}


	/** Creates a new instance using a specific initial permutation.
	 *
	 * @param symGraph a symmetric, loopless graph.
	 * @param startPerm an initial permutation of the graph, or {@code null} for no permutation.
	 * @param seed a random seed.
	 */
	public LayeredLabelPropagation(final ImmutableGraph symGraph, final long[][] startPerm, final long seed) throws IOException {
		this(symGraph, startPerm, seed, false);
	}

	/** Creates a new instance using a specific initial permutation.
	 *
	 * <p>If <code>exact</code> is true, the final permutation is
	 * <em>exactly</em> the same as if you first permute the graph with <code>startPerm</code> and
	 * then apply LLP with an {@code null} starting permutation.
	 *
	 * @param symGraph a symmetric, loopless graph.
	 * @param startPerm an initial permutation of the graph, or {@code null} for no permutation.
	 * @param seed a random seed.
	 * @param exact a boolean flag that forces the algorithm to run exactly.
	 */
	public LayeredLabelPropagation(final ImmutableGraph symGraph, final long[][] startPerm, final long seed, final boolean exact) throws IOException {
		this(symGraph, startPerm, 0, seed, exact);
	}

	/** Creates a new instance using a specific initial permutation and specified number of threads.
	 *
	 * <p>If <code>exact</code> is true, the final permutation is
	 * <em>exactly</em> the same as if you first permute the graph with <code>startPerm</code> and
	 * then apply LLP with an {@code null} starting permutation.
	 *
	 * @param symGraph a symmetric, loopless graph.
	 * @param startPerm an initial permutation of the graph, or {@code null} for no permutation.
	 * @param numberOfThreads the number of threads to be used (0 for automatic sizing).
	 * @param seed a random seed.
	 * @param exact a boolean flag that forces the algorithm to run exactly.
	 */
	public LayeredLabelPropagation(final ImmutableGraph symGraph, final long[][] startPerm, final int numberOfThreads, final long seed, final boolean exact) throws IOException {
		this.symGraph = symGraph;
		this.n = symGraph.numNodes();
		this.startPerm = startPerm;
		this.seed = seed;
		this.r = new XoRoShiRo128PlusRandom(seed);
		this.exact = exact;
		this.label = LongBigArrays.newBigAtomicArray(n);
		this.volume = LongBigArrays.newBigAtomicArray(n);
		cumulativeOutdegrees = new EliasFanoCumulativeOutdegreeList(symGraph, symGraph.numArcs(), 1);

		this.gapCost = new MutableDouble();
		this.updateList = Util.identity(n);
		simpleUncaughtExceptionHandler = new SimpleUncaughtExceptionHandler();
		labelling = File.createTempFile(this.getClass().getName(), "labelling");
		labelling.deleteOnExit();

		this.numberOfThreads = numberOfThreads != 0 ? numberOfThreads : Runtime.getRuntime().availableProcessors();
		this.canChange = BooleanBigArrays.newBigArray(n);
		this.modified = new AtomicLong(0);
		this.objectiveFunction = new double[this.numberOfThreads];
	}


	/**
	 * Sets the basename for label files.
	 *
	 * @param labelBasename basename for label files.
	 */
	public void labelBasename(final String labelBasename) {
		labelBasenameSet = true;
		labelling = new File(labelBasename);
	}


	/**
	 * Combines two labellings devilishly into a new one.
	 *
	 * @param label the minor label; the result will be stored here.
	 * @param major the major label.
	 * @param perm a virtual permutation applied to the graph, or {@code null} for no permutation.
	 * @param support a support array.
	 * @return the resulting number of labels.
	 */
	private static long combine(final long[][] label, final long[][] major, final long[][] perm, final long[][] support) {
		final long n = length(label);
		if (n == 0) return 0;
		if (n != length(major)) throw new IllegalArgumentException();

		Util.identity(support);

		if (perm == null) LongBigArrays.parallelQuickSort(support, (a, b) -> {
			final long majorA = get(major, a);
			final long majorB = get(major, b);
			int t = Long.compare(get(label, majorA), get(label, majorB));
			if (t != 0) return t;
			t = Long.compare(majorA, majorB);
			if (t != 0) return t;
			t = Long.compare(get(label, a), get(label, b));
			if (t != 0) return t;
			return Long.compare(a, b);
		});
		else LongBigArrays.parallelQuickSort(support, (a, b) -> {
			final long majorA = get(major, a);
			final long majorB = get(major, b);
			int t = Long.compare(get(label, majorA), get(label, majorB));
			if (t != 0) return t;
			t = Long.compare(get(perm, majorA), get(perm, majorB));
			if (t != 0) return t;
			t = Long.compare(get(label, a), get(label, b));
			if (t != 0) return t;
			return Long.compare(a, b);
		});

		long currMinor = get(label, get(support, 0));
		long currMajor = get(major, get(support, 0));
		long curr = 0;
		set(label, get(support, 0), curr);

		for (long i = 1; i < n; i++) {
			final long t = get(support, i);
			final long u = get(label, t);
			if (get(major, t) != currMajor || u != currMinor) {
				currMinor = u;
				currMajor = get(major, t);
				curr++;
			}

			set(label, t, curr);
		}

		return ++curr;

	}

	/** A minimal implementation of a set of counters using a hash table without rehashing. */
	private final static class OpenHashTableCounter {
		/** The keys. Always sized as a power of two. */
		private long[] key;
		/** The counters associated to {@link #key}. */
		private int[] count;
		/** Keeps track of the location of each key. Useful for linear-time iteration over the key/value pairs. */
		private int[] location;
		/** The mask used to compute the key locations. */
		private int mask;
		/** The number of keys in the table. */
		private int n;

		public OpenHashTableCounter() {
			mask = -1;
			count = IntArrays.EMPTY_ARRAY;
			key = LongArrays.EMPTY_ARRAY;
			location = IntArrays.EMPTY_ARRAY;
		}

		public void incr(final long k) {
			int pos = (int)((k * 2056437379) & mask);
			while (count[pos] != 0 && key[pos] != k)
				pos = (pos + 1) & mask;
			if (count[pos]++ == 0) {
				key[pos] = k;
				location[n++] = pos;
			}
		}

		public boolean containsKey(final long k) {
			int pos = (int)((k * 2056437379) & mask);
			while (count[pos] != 0 && key[pos] != k)
				pos = (pos + 1) & mask;
			return count[pos] != 0;
		}

		// After a call to this method, incr() cannot be called anymore.
		public void addZeroCount(final long k) {
			int pos = (int)((k * 2056437379) & mask);
			while (count[pos] != 0 && key[pos] != k)
				pos = (pos + 1) & mask;
			if (count[pos] == 0) {
				key[pos] = k;
				location[n++] = pos;
			}
		}

		private final static class Entry extends AbstractLong2IntMap.BasicEntry {
			public Entry() {
				super(0, 0);
			}

			public void setKey(final long key) {
				this.key = key;
			}

			@Override
			public int setValue(final int value) {
				this.value = value;
				return -1; // Violates the interface, but it's all internal.
			}
		}

		public Iterator<Long2IntMap.Entry> entries() {
			return new ObjectIterator<Long2IntMap.Entry>() {
				private int i;

				private final Entry entry = new Entry();

				@Override
				public boolean hasNext() {
					return i < n;
				}

				@Override
				public Entry next() {
					if (!hasNext()) throw new NoSuchElementException();
					final int l = location[i++];
					entry.setKey(key[l]);
					entry.setValue(count[l]);
					return entry;
				}
			};
		}

		public void clear(final int size) {
			if (mask + 1 < (1 << (Fast.ceilLog2(size) + 1))) {
				mask = (1 << (Fast.ceilLog2(size) + 1)) - 1;
				count = new int[mask + 1];
				key = new long[mask + 1];
				location = new int[mask + 1];
			}
			else while (n-- != 0) count[location[n]] = 0;
			n = 0;
		}
	}

	private final class GapCostThread extends Thread {
		@SuppressWarnings("hiding")
		private final ImmutableGraph symGraph;

		/** The permutation whose cost is to be evaluated. */
		private final long[][] perm;

		private GapCostThread(final ImmutableGraph symGraph, final long[][] perm) {
			this.symGraph = symGraph;
			this.perm = perm;
		}

		@Override
		public void run() {
			final ImmutableGraph symGraph = this.symGraph;
			final long numNodes = LayeredLabelPropagation.this.n;
			final long numArcs = LayeredLabelPropagation.this.symGraph.numArcs();
			final long[][] perm = this.perm;
			long[][] permutedSuccessors = LongBigArrays.newBigArray(32);
			long[][] successors;
			final long granularity = Math.max(1024, numArcs >>> 9);
			long start;
			long end;

			double gapCost = 0;
			for (;;) {

				// Try to get another piece of work.
				synchronized(LayeredLabelPropagation.this.cumulativeOutdegrees) {
					if (nextNode == numNodes) {
						LayeredLabelPropagation.this.gapCost.add(gapCost);
						break;
					}
					start = nextNode;
					final long target = nextArcs + granularity;
					if (target >= numArcs) nextNode = numNodes;
					else {
						nextArcs = cumulativeOutdegrees.skipTo(target);
						nextNode = cumulativeOutdegrees.currentIndex();
					}
					end = nextNode;
				}

				final NodeIterator nodeIterator = symGraph.nodeIterator(start);
				for (long i = start; i < end; i++) {
					nodeIterator.nextLong();
					final long node = get(perm, i);
					final long outdegree = nodeIterator.outdegree();
					if (outdegree > 0) {
						successors = nodeIterator.successorBigArray();
						permutedSuccessors = grow(permutedSuccessors, outdegree);
						for (long j = outdegree; j-- != 0;)
							set(permutedSuccessors, j, get(perm, get(successors, j)));
						LongBigArrays.quickSort(permutedSuccessors, 0, outdegree);
						long prev = node;
						for (long j = 0; j < outdegree; j++) {
							final long curr = get(permutedSuccessors, j);
							gapCost += Fast.ceilLog2(Math.abs(prev - curr));
							prev = curr;
						}
					}
				}
			}
		}
	}

	private final class IterationThread extends Thread {
		@SuppressWarnings("hiding")
		private final ImmutableGraph symGraph;

		/** The current value of &gamma;. */
		private final double gamma;

		/** A progress logger. */
		private final ProgressLogger pl;

		private final int index;

		private IterationThread(final ImmutableGraph symGraph, final double gamma, final int index, final ProgressLogger pl) {
			this.symGraph = symGraph;
			this.gamma = gamma;
			this.index = index;
			this.pl = pl;
		}

		@Override
		public void run() {
			final XoRoShiRo128PlusRandom r = new XoRoShiRo128PlusRandom(LayeredLabelPropagation.this.seed);
			final AtomicLongArray[] label = LayeredLabelPropagation.this.label;
			final AtomicLongArray[] volume = LayeredLabelPropagation.this.volume;
			final ImmutableGraph symGraph = this.symGraph;
			final long numNodes = LayeredLabelPropagation.this.n;
			final long numArcs = LayeredLabelPropagation.this.symGraph.numArcs();
			final long[][] updateList = LayeredLabelPropagation.this.updateList;
			final long[][] startPerm = LayeredLabelPropagation.this.startPerm;
			final boolean[][] canChange = LayeredLabelPropagation.this.canChange;
			final boolean exact = LayeredLabelPropagation.this.exact;
			final double gamma = this.gamma;
			final long granularity = Math.max(1024, numArcs >>> 9);

			long start;
			long end;
			double delta = LayeredLabelPropagation.this.objectiveFunction[index];

			for (;;) {

				// Try to get another piece of work.
				synchronized(LayeredLabelPropagation.this.cumulativeOutdegrees) {
					if (nextNode == numNodes) {
						LayeredLabelPropagation.this.objectiveFunction[index] = delta;
						break;
					}
					start = nextNode;
					final long target = nextArcs + granularity;
					if (target >= numArcs) nextNode = numNodes;
					else {
						nextArcs = cumulativeOutdegrees.skipTo(target);
						nextNode = cumulativeOutdegrees.currentIndex();
					}
					end = nextNode;
				}

				final OpenHashTableCounter map = new OpenHashTableCounter();

				for (long i = start; i < end; i++) {
					final long node = get(updateList, i);

					/** Note that here we are using a heuristic optimisation: if no neighbour has changed,
					 *  the label of a node cannot change. If gamma != 0, this is not necessarily true,
					 *  as a node might need to change its value just because of a change of volume of
					 *  the adjacent labels. */

					if (get(canChange, node)) {
						set(canChange, node, false);
						final long longOutdegree = symGraph.outdegree(node);
						if (longOutdegree > Integer.MAX_VALUE) throw new IllegalArgumentException("This implementation supports outdegree at most " + Integer.MAX_VALUE);
						final int outdegree = (int)longOutdegree;
						if (outdegree > 0) {
							final long currentLabel = get(label, node);
							decrementAndGet(volume, currentLabel);

							map.clear(outdegree);
							LazyLongIterator successors = symGraph.successors(node);
							for (long j = outdegree; j-- != 0;) map.incr(get(label, successors.nextLong()));

							if (!map.containsKey(currentLabel)) map.addZeroCount(currentLabel);

							double max = Double.NEGATIVE_INFINITY;
							double old = 0;
							final LongArrayList majorities = new LongArrayList();

							for (final Iterator<Long2IntMap.Entry> entries = map.entries(); entries.hasNext();) {
								final Long2IntMap.Entry entry = entries.next();
								final long l = entry.getLongKey();
								final int freq = entry.getIntValue(); // Frequency of label in my
								// neighbourhood
								final double val = freq - gamma * (get(volume, l) + 1 - freq);

								if (max == val) majorities.add(l);

								if (max < val) {
									majorities.clear();
									max = val;
									majorities.add(l);
								}

								if (l == currentLabel) old = val;
							}

							if (exact) {
								if (startPerm != null) LongArrays.quickSort(majorities.elements(), 0, majorities.size(), (a, b) -> Long.compare(get(startPerm, a), get(startPerm, b)));
								else LongArrays.quickSort(majorities.elements(), 0, majorities.size());
							}


							// Extract a label from the majorities
							final long nextLabel = majorities.getLong(r.nextInt(majorities.size()));
							if (nextLabel != currentLabel) {
								modified.addAndGet(1);
								successors = symGraph.successors(node);
								for (long j = outdegree; j-- != 0;) set(canChange, successors.nextLong(), true);
							}
							set(label, node, nextLabel);
							incrementAndGet(volume, nextLabel);

							delta += max - old;
						}
					}
				}
				synchronized (pl) {
					pl.update(end - start);
				}
			}
		}
	}



	private final class SimpleUncaughtExceptionHandler implements UncaughtExceptionHandler {
		@Override
		public void uncaughtException(final Thread t, final Throwable e) {
			threadException = e;
		}
	}

	private void update(final double gamma) {
		final long n = this.n;
		final long[][] updateList = this.updateList;
		modified.set(0);
		nextArcs = nextNode = 0;

		if (exact) {
			if (startPerm == null) Util.identity(updateList);
			else Util.invertPermutation(startPerm, updateList);
		}

		// Local shuffle
		for (long i = 0; i < n;) LongBigArrays.shuffle(updateList, i, Math.min(i += SHUFFLE_GRANULARITY, n), r);

		final ProgressLogger pl = new ProgressLogger(LOGGER);
		pl.expectedUpdates = n;
		pl.logInterval = ProgressLogger.TEN_SECONDS;
		pl.itemsName = "nodes";
		pl.start("Starting update " + update + "...");

		final Thread[] thread = new Thread[numberOfThreads];

		nextArcs = nextNode =  0;
		for (int i = 0; i < numberOfThreads; i++) {
			thread[i] = new IterationThread(symGraph.copy(), gamma, i, pl);
			thread[i].setUncaughtExceptionHandler(simpleUncaughtExceptionHandler);
			thread[i].start();
		}

		for (int i = 0; i < numberOfThreads; i++)
			try {
				thread[i].join();
			}
			catch (final InterruptedException e) {
				throw new RuntimeException(e);
			}

		if (threadException != null) throw new RuntimeException(threadException);
		pl.done();
	}



	private void computeGapCost(final long[][] newPerm) {
		final long[][] startPerm = this.startPerm;
		final AtomicLongArray[] label = this.label;

		Util.identity(newPerm);
		if (startPerm != null) LongBigArrays.quickSort(newPerm, (x, y) -> {
			final int t = Long.compare(get(startPerm, get(label, x)), get(startPerm, get(label, y)));
			return t != 0 ? t : Long.compare(get(startPerm, x), get(startPerm, y));
		});
		else LongBigArrays.quickSort(newPerm, (x, y) -> {
			final int t = Long.compare(get(label, x), get(label, y));
			return t != 0 ? t : Long.compare(x, y);
		});

		Util.invertPermutationInPlace(newPerm);

		final Thread[] thread = new Thread[numberOfThreads];

		nextArcs = nextNode =  0;
		for (int i = 0; i < numberOfThreads; i++) (thread[i] = new GapCostThread(symGraph.copy(), newPerm)).start();

		for (int i = 0; i < numberOfThreads; i++)
			try {
				thread[i].join();
			}
			catch (final InterruptedException e) {
				throw new RuntimeException(e);
			}
	}


	private double objectiveFunction() {
		double res = 0;
		for (final double d : objectiveFunction) res += d;
		return res;
	}

	private void init() {
		long p = 0;
		for (int s = 0; s < label.length; s++) {
			final AtomicLongArray t = label[s];
			final AtomicLongArray u = volume[s];
			final int l = t.length();
			for (int d = 0; d < l; d++, p++) {
				t.set(d, p);
				u.set(d, 1);
			}
		}
		Util.identity(updateList);
		BigArrays.fill(canChange, true);
		Arrays.fill(objectiveFunction, 0.);
	}

	/**
	 * Computes the labels of a graph for a given value of &gamma; using the {@linkplain #MAX_UPDATES default maximum number of updates}.
	 *
	 * @param gamma the gamma parameter.
	 * @return the labels.
	 */
	public AtomicLongArray[] computeLabels(final double gamma) {
		return computeLabels(gamma, MAX_UPDATES);
	}
	/**
	 * Computes the labels of a graph for a given value of &gamma;.
	 *
	 * @param gamma the gamma parameter.
	 * @param maxUpdates the maximum number of updates performed.
	 * @return the labels.
	 */
	public AtomicLongArray[] computeLabels(final double gamma, final int maxUpdates) {
		init();
		final String gammaFormatted = GAMMA_FORMAT.format(gamma);
		double prevObjFun = 0;
		double gain = 0;
		final ProgressLogger pl = new ProgressLogger(LOGGER, "updates");
		pl.logger().info("Running " + this.numberOfThreads + " threads");
		pl.start("Starting iterations with gamma=" + gammaFormatted + "...");

		update = 0;

		do {
			prevObjFun = objectiveFunction();
			update(gamma);
			pl.updateAndDisplay();
			gain = 1 - (prevObjFun / objectiveFunction());
			LOGGER.info("Gain: " + gain);
			LOGGER.info("Modified: " + modified.get());
			update++;
		} while (modified.get() > 0 && gain > GAIN_TRESHOLD && update < maxUpdates);

		pl.done();

		return label;
	}

	/**
	 * Computes the final permutation of the graph  using the {@linkplain #MAX_UPDATES default maximum number of updates} and
	 * the {@linkplain #DEFAULT_GAMMAS default gammas}.
	 *
	 * @param cluster if not {@code null}, clusters will be saved to a file with this name.
	 * @return the final permutation of the graph.
	 */
	public long[][] computePermutation(final String cluster) throws IOException {
		return computePermutation(DEFAULT_GAMMAS, cluster, MAX_UPDATES);
	}

	/**
	 * Computes the final permutation of the graph  using the {@linkplain #MAX_UPDATES default maximum number of updates}.
	 *
	 * @param gammas a set of parameters that will be used to generate labellings.
	 * @param cluster if not {@code null}, clusters will be saved to a file with this name.
	 * @return the final permutation of the graph.
	 */
	public long[][] computePermutation(final double[] gammas, final String cluster) throws IOException {
		return computePermutation(gammas, cluster, MAX_UPDATES);
	}

	/**
	 * Computes the final permutation of the graph.
	 *
	 * @param gammas a set of parameters that will be used to generate labellings.
	 * @param cluster if not {@code null}, clusters will be saved to a file with this name.
	 * @param maxUpdates the maximum number of updates performed.
	 * @return the final permutation of the graph.
	 */
	public long[][] computePermutation(final double[] gammas, final String cluster, final int maxUpdates) throws IOException {
		final long n = this.n;
		final int m = gammas.length;

		final double[] gapCosts = new double[m];

		final ProgressLogger plGammas = new ProgressLogger(LOGGER);
		plGammas.itemsName = "gammas";
		plGammas.expectedUpdates = m;
		plGammas.start();

		for (int index = 0; index < m; index++) {
			init();
			final double gamma = gammas[index];
			final String gammaFormatted = GAMMA_FORMAT.format(gamma);
			double prevObjFun = 0;
			double gain = 0;

			final ProgressLogger pl = new ProgressLogger(LOGGER, "updates");
			pl.logger().info("Running " + this.numberOfThreads + " threads");
			pl.start("Starting iterations with gamma=" + gammaFormatted + " (" + (index + 1) + "/" + m + ") ...");

			update = 0;

			do {
				prevObjFun = objectiveFunction();
				update(gamma);
				pl.updateAndDisplay();
				gain = 1 - (prevObjFun / objectiveFunction());
				LOGGER.info("Gain: " + gain);
				LOGGER.info("Modified: " + modified.get());
				update++;
			} while (modified.get() > 0 && gain > GAIN_TRESHOLD && update < maxUpdates);

			pl.done();

			final long length = length(label);
			final DataOutputStream dos = new DataOutputStream(new FastBufferedOutputStream(new FileOutputStream(labelling + "-" + index)));
			for (long i = 0; i < length; i++)
				dos.writeLong(get(label, i));
			dos.close();

			if (!labelBasenameSet) new File(labelling + "-" + index).deleteOnExit();


			gapCost.setValue(0);

			computeGapCost(updateList);
			gapCosts[index] = gapCost.doubleValue();
			LOGGER.info("Completed iteration with gamma " + gammaFormatted + " (" + (index + 1) + "/" + m + ") , gap cost: " + gapCost.doubleValue());
			plGammas.updateAndDisplay();
		}
		plGammas.done();

		label = null; // We no longer need the atomic list

		final int[] best = Util.identity(m);
		IntArrays.quickSort(best, 0, best.length, (x, y) -> Double.compare(gapCosts[y], gapCosts[x]));

		final int bestGamma = best[m - 1];
		LOGGER.info("Best gamma: " + GAMMA_FORMAT.format(gammas[bestGamma]) + "\twith GapCost: " + gapCosts[bestGamma]);
		LOGGER.info("Worst gamma: " + GAMMA_FORMAT.format(gammas[best[0]]) + "\twith GapCost: " + gapCosts[best[0]]);


		final long intLabel[][] = BinIO.loadLongsBig(labelling + "-" + bestGamma);
		if (startPerm != null) for (long i = 0; i < n; i++) set(intLabel, i, get(startPerm, get(intLabel, i)));


		for (int step = 0; step < m; step++) {
			LOGGER.info("Starting step " + step + "...");
			long[][] major = BinIO.loadLongsBig(labelling + "-" + best[step]);
			combine(intLabel, major, startPerm, updateList);
			major = BinIO.loadLongsBig(labelling + "-" + bestGamma);
			final long numberOflabels = combine(intLabel, major, startPerm, updateList);
			LOGGER.info("Number of labels: " + numberOflabels);
			LOGGER.info("Finished step " + step);
		}


		final long[][] newPerm = this.updateList; // It is no longer necessary: we reuse it.
		final long[][] startPerm = this.startPerm;
		Util.identity(newPerm);
		if (startPerm == null) BigArrays.mergeSort(0, length(newPerm), (x, y) -> {
			final long newPermX = get(newPerm, x);
			final long newPermY = get(newPerm, y);
			final int t = Long.compare(get(intLabel, newPermX), get(intLabel, newPermY));
			return t != 0 ? t : Long.compare(newPermX, newPermY);
		}, (x, y) -> swap(newPerm, x, y));
		else BigArrays.mergeSort(0, length(newPerm), (x, y) -> {
			final long newPermX = get(newPerm, x);
			final long newPermY = get(newPerm, y);
			final int t = Long.compare(get(intLabel, newPermX), get(intLabel, newPermY));
			return t != 0 ? t : Long.compare(get(startPerm, newPermX), get(startPerm, newPermY));
		}, (x, y) -> swap(newPerm, x, y));

		if (cluster != null) {
			final DataOutputStream dos = new DataOutputStream(new FastBufferedOutputStream(new FileOutputStream(cluster)));

			// Printing clusters; volume is really the best saved clustering
			BinIO.loadLongs(labelling + "-" + bestGamma, intLabel);
			long current = get(intLabel, get(newPerm, 0));
			long j = 0;
			for (long i = 0; i < n; i++) {
				final long tmp = get(intLabel, get(newPerm, i));
				if (tmp != current) {
					current = tmp;
					j++;
				}
				dos.writeLong(j);
			}
			dos.close();
		}

		Util.invertPermutationInPlace(newPerm);

		return newPerm;
	}


	public static void main(final String[] args) throws IOException, JSAPException {
		final SimpleJSAP jsap = new SimpleJSAP(LayeredLabelPropagation.class.getName(), "Runs the Layered Label Propagation algorithm on a graph.", new Parameter[] {
				new FlaggedOption("gammas", JSAP.STRING_PARSER, "-0,-1,-2,-3,-4,-5,-6,-7,-8,-9,-10,0-0", JSAP.NOT_REQUIRED, 'g', "gammas",
						"The set of values of gamma, expressed as a comma-separated list of dyadics k/2^j specified as [k]-j (if missing, k=1)."),
				new FlaggedOption("threads", JSAP.INTSIZE_PARSER, "0", JSAP.NOT_REQUIRED, 'T', "threads", "The number of threads to be used. If 0, the number will be estimated automatically."),
				new FlaggedOption("maxUpdates", JSAP.INTEGER_PARSER, Integer.toString(MAX_UPDATES), JSAP.NOT_REQUIRED, 'u', "max-updates", "Maximum number of updates."),
				new FlaggedOption("cluster", JSAP.STRING_PARSER, null, JSAP.NOT_REQUIRED, 'c', "clusters", "Store clusters id in the given file."),
				new Switch("random", 'r', "random", "The graph will be virtually permuted in a random fashion."),
				new FlaggedOption("randomSeed", JSAP.LONG_PARSER, JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, 's', "random-seed", "The random seed."),
				new FlaggedOption("labelBasename", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, 'l', "label-basename", "A basename for label files."),
				new Switch("mapped", 'm', "mapped", "The graph will be mapped into memory, rather than loaded."),
				new UnflaggedOption("symGraph", JSAP.STRING_PARSER, JSAP.REQUIRED, "The basename of a symmetric, loopless version of the graph."),
				new UnflaggedOption("perm", JSAP.STRING_PARSER, JSAP.REQUIRED, "The output permutation."), });

		final JSAPResult jsapResult = jsap.parse(args);
		if (jsap.messagePrinted()) System.exit(1);

		final boolean mapped = jsapResult.getBoolean("mapped");
		final boolean random = jsapResult.getBoolean("random");
		final int threads = jsapResult.getInt("threads");

		final ImmutableGraph symGraph = mapped ? ImmutableGraph.loadMapped(jsapResult.getString("symGraph")) : ImmutableGraph.load(jsapResult.getString("symGraph"));
		final long n = symGraph.numNodes();

		final long[][] startPerm = mapped && !random ? null : Util.identity(n);
		final XoRoShiRo128PlusRandom r = jsapResult.userSpecified("randomSeed") ? new XoRoShiRo128PlusRandom(jsapResult.getLong("randomSeed")) : new XoRoShiRo128PlusRandom();
		if (random) LongBigArrays.shuffle(startPerm, r);

		// TODO
		// if (startPerm != null && ! mapped) startPerm =
		// Util.invertPermutationInPlace(DFS.dfsperm(symGraph, startPerm));

		final LayeredLabelPropagation clustering = new LayeredLabelPropagation(symGraph, startPerm, threads, jsapResult.userSpecified("randomSeed") ? jsapResult.getLong("randomSeed") : r.nextLong(), false);
		if (jsapResult.userSpecified("labelBasename")) clustering.labelBasename(jsapResult.getString("labelBasename"));

		final DoubleArrayList gammas = new DoubleArrayList();
		for (final String gamma : jsapResult.getString("gammas").split(",")) {
			final String[] p = gamma.split("-");
			gammas.add((p[0].length() != 0 ? Integer.parseInt(p[0]) : 1) * Math.pow(1. / 2, Integer.parseInt(p[1])));
		}
		Collections.sort(gammas);
		final long[][] permutation = clustering.computePermutation(gammas.toDoubleArray(), jsapResult.getString("cluster"), jsapResult.getInt("maxUpdates"));
		BinIO.storeLongs(permutation, jsapResult.getString("perm"));
	}
}
