package it.unimi.dsi.law.big.graph;

/*
 * Copyright (C) 2010-2020 Paolo Boldi, Massimo Santini and Sebastiano Vigna
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 3 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 */

import java.io.File;
import java.io.IOException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.primitives.Longs;
import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.SimpleJSAP;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;

import it.unimi.dsi.Util;
import it.unimi.dsi.big.webgraph.ImmutableGraph;
import it.unimi.dsi.big.webgraph.LazyLongIterator;
import it.unimi.dsi.bits.LongArrayBitVector;
import it.unimi.dsi.fastutil.Arrays;
import it.unimi.dsi.fastutil.BigArrays;
import it.unimi.dsi.fastutil.io.BinIO;
import it.unimi.dsi.fastutil.longs.LongArrayList;
import it.unimi.dsi.fastutil.longs.LongArrays;
import it.unimi.dsi.fastutil.longs.LongBigArrays;
import it.unimi.dsi.io.ByteDiskQueue;
import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.util.XoRoShiRo128PlusRandom;

/**
 * Computes the visit order with respect to a breadth-first visit.
 *
 * @author Marco Rosa
 * @author Thibault Allançon
 */

// RELEASE-STATUS: DIST

public class BFS {
	private final static Logger LOGGER = LoggerFactory.getLogger(BFS.class);

	/**
	 * Return the permutation induced by the visit order of a breadth-first visit.
	 *
	 * @param graph
	 *            a graph.
	 * @param startingNode
	 *            the only starting node of the visit, or -1 for a complete
	 *            visit.
	 * @param startPerm
	 *            a permutation that will be used to shuffle successors, or {@code null}.
	 * @param bufferSize
	 *            internal queue buffer size (will be minimized with {@link Long#SIZE} times the number of nodes in the graph).
	 * @return the permutation induced by the visit order of a depth-first
	 *         visit.
	 */
	public static long[][] bfsperm(final ImmutableGraph graph, final long startingNode, final long[][] startPerm, final int bufferSize) throws IOException {
		return bfsperm(graph, startingNode, startPerm, bufferSize, true);
	}

	/**
	 * Return the permutation induced by the visit order of a breadth-first visit.
	 *
	 * @param graph
	 *            a graph.
	 * @param startingNode
	 *            the only starting node of the visit, or -1 for a complete
	 *            visit.
	 * @param bufferSize
	 *            internal queue buffer size (will be minimized with {@link Long#SIZE} times the number of nodes in the graph).
	 * @return the permutation induced by the visit order of a depth-first
	 *         visit.
	 */
	public static long[][] bfsperm(final ImmutableGraph graph, final long startingNode, final int bufferSize) throws IOException {
		return bfsperm(graph, startingNode, null, bufferSize, true);
	}

	/**
	 * Performs a breadth-first visit.
	 *
	 * @param graph
	 *            a graph.
	 * @param startingNode
	 *            the only starting node of the visit, or -1 for a complete
	 *            visit.
	 * @param bufferSize
	 *            internal queue buffer size (will be minimized with {@link Long#SIZE} times the number of nodes in the graph).
	 * @return the permutation induced by the visit order of a depth-first
	 *         visit.
	 */
	public static long[][] bfs(final ImmutableGraph graph, final long startingNode, final int bufferSize) throws IOException {
		return bfsperm(graph, startingNode, null, bufferSize, false);
	}

	/**
	 * Performs a breadth-first visit.
	 *
	 * @param graph
	 *            a graph.
	 * @param startingNode
	 *            the only starting node of the visit, or -1 for a complete
	 *            visit.
	 * @param startPerm
	 *            a permutation that will be used to shuffle successors, or {@code null}.
	 * @param bufferSize
	 *            internal queue buffer size (will be minimized with {@link Long#SIZE} times the number of nodes in the graph).
	 * @return the permutation induced by the visit order of a depth-first
	 *         visit.
	 */
	public static long[][] bfs(final ImmutableGraph graph, final long startingNode, final long[][] startPerm, final int bufferSize) throws IOException {
		return bfsperm(graph, startingNode, startPerm, bufferSize, false);
	}

	/**
	 * Return the permutation induced by the visit order of a breadth-first visit.
	 *
	 * @param graph
	 *            a graph.
	 * @param startingNode
	 *            the only starting node of the visit, or -1 for a complete
	 *            visit.
	 * @param startPerm
	 *            a permutation that will be used to shuffle successors, or {@code null}.
	 * @param bufferSize
	 *            internal queue buffer size (will be minimized with {@link Long#SIZE} times the number of nodes in the graph).
	 * @param doPerm if {@code true}, returns a permutation; otherwise, {@code null}.
	 * @return the permutation induced by the visit order of a depth-first
	 *         visit.
	 */
	public static long[][] bfsperm(final ImmutableGraph graph, final long startingNode, final long[][] startPerm, int bufferSize, final boolean doPerm) throws IOException {
		final long n = graph.numNodes();
		// Allow enough memory to behave like in-memory queue
		bufferSize = Math.min(bufferSize, (int)Math.min(Arrays.MAX_ARRAY_SIZE & ~0x7, 8L * n));

		final long[][] visitOrder = doPerm ? LongBigArrays.newBigArray(n) : null;
		if (doPerm) BigArrays.fill(visitOrder, -1);
		final long[][] invStartPerm = startPerm == null ? null : Util.invertPermutation(startPerm, LongBigArrays.newBigArray(n));
		// Use a disk based queue to store BFS frontier
		final File queueFile = File.createTempFile(BFS.class.getSimpleName(), "queue");
		final ByteDiskQueue queue = ByteDiskQueue.createNew(queueFile, bufferSize, true);
		final byte[] byteBuf = new byte[Long.BYTES];
		// WARNING: no 64-bit version of this data-structure, but it can support
		// indices up to 2^37
		final LongArrayBitVector visited = LongArrayBitVector.ofLength(n);
		final ProgressLogger pl = new ProgressLogger(LOGGER);
		pl.expectedUpdates = n;
		pl.itemsName = "nodes";
		pl.start("Starting breadth-first visit...");

		long pos = 0;

		if (startPerm != null) {
			for (long i = 0; i < n; i++) {
				final long start = i == 0 && startingNode != -1 ? startingNode : BigArrays.get(invStartPerm, i);
				if (visited.getBoolean(start)) continue;
				queue.enqueue(Longs.toByteArray(start));
				visited.set(start);

				final LongArrayList successors = new LongArrayList();

				while (!queue.isEmpty()) {
					queue.dequeue(byteBuf);
					final long currentNode = Longs.fromByteArray(byteBuf);

					if (doPerm) BigArrays.set(visitOrder, pos, currentNode);
					pos++;
					final LazyLongIterator iterator = graph.successors(currentNode);

					successors.clear();
					long succ;
					while((succ = iterator.nextLong()) != -1) {
						if (!visited.getBoolean(succ)) {
							successors.add(succ);
							visited.set(succ);
						}
					}

					final long[] randomSuccessors = successors.elements();
					LongArrays.quickSort(randomSuccessors, 0, successors.size(), (x, y) -> {
						return Long.compare(BigArrays.get(startPerm, y), BigArrays.get(startPerm, x));
					});

					for (int j = successors.size(); j-- != 0;)
						queue.enqueue(Longs.toByteArray(randomSuccessors[j]));
					pl.update();
				}

				if (startingNode != -1) break;
			}
		}
		else {
			for (long i = 0; i < n; i++) {
				final long start = i == 0 && startingNode != -1 ? startingNode : i;
				if (visited.getBoolean(start)) continue;
				queue.enqueue(Longs.toByteArray(start));
				visited.set(start);

				while (!queue.isEmpty()) {
					queue.dequeue(byteBuf);
					final long currentNode = Longs.fromByteArray(byteBuf);

					if (doPerm) BigArrays.set(visitOrder, pos, currentNode);
					pos++;
					final LazyLongIterator iterator = graph.successors(currentNode);
					long succ;
					while((succ = iterator.nextLong()) != -1) {
						if (!visited.getBoolean(succ)) {
							visited.set(succ);
							queue.enqueue(Longs.toByteArray(succ));
						}
					}

					pl.update();
				}

				if (startingNode != -1) break;
			}

		}
		pl.done();
		queue.close();
		return visitOrder;
	}

	public static void main(final String[] args) throws JSAPException, IOException {
		final SimpleJSAP jsap = new SimpleJSAP(BFS.class.getName(), "Computes the permutation induced by a breadth-first visit.",
			new Parameter[] {
				new FlaggedOption("randomSeed", JSAP.LONG_PARSER, "0", JSAP.NOT_REQUIRED, 'r', "random-seed", "The random seed."),
				new FlaggedOption("initialNode", JSAP.LONG_PARSER, "-1", JSAP.NOT_REQUIRED, 'i', "initial-node", "The initial node of the visit. If specified, the visit will be performed only starting from the given node. The default performs a complete visit, iterating on all possible starting nodes."),
				new FlaggedOption("bufferSize", JSAP.INTEGER_PARSER, String.valueOf(Arrays.MAX_ARRAY_SIZE & ~0x7), JSAP.NOT_REQUIRED, 's', "buffer-size", "Internal queue buffer size."),
				new Switch("mapped", 'm', "mapped", "Use memory-mapping."),
				new Switch("random", 'p', "Start from a random permutation."),
				new UnflaggedOption("graph", JSAP.STRING_PARSER, JSAP.REQUIRED, "The basename of the input graph"),
				new UnflaggedOption("perm", JSAP.STRING_PARSER, JSAP.REQUIRED, "The name of the output permutation, or - for no permutation (e.g., for benchmarking)."),
		});

		final JSAPResult jsapResult = jsap.parse(args);
		if (jsap.messagePrinted()) System.exit(1);

		final String basename = jsapResult.getString("graph");
		final ImmutableGraph graph = jsapResult.userSpecified("mapped") ? ImmutableGraph.loadMapped(basename) : ImmutableGraph.load(basename);

		final long n = graph.numNodes();
		final long seed = jsapResult.getLong("randomSeed");
		long[][] startPerm = null;
		if (jsapResult.getBoolean("random")) {
			startPerm = Util.identity(LongBigArrays.newBigArray(n));
			LongBigArrays.shuffle(startPerm, new XoRoShiRo128PlusRandom(seed));
		}
		final long initialnode = jsapResult.getLong("initialNode");
		final int bufferSize = jsapResult.getInt("bufferSize");

		final String perm = jsapResult.getString("perm");
		if ("-".equals(perm)) bfs(graph, initialnode, startPerm, bufferSize);
		else BinIO.storeLongs(Util.invertPermutationInPlace(bfsperm(graph, initialnode, startPerm, bufferSize)), perm );
	}
}
