package voting;

/*		 
 * Voting in social networks
 *
 * Copyright (C) 2009-2010 Paolo Boldi, Francesco Bonchi, Carlos Castillo, Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 2.1 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


import it.unimi.dsi.Util;
import it.unimi.dsi.fastutil.ints.IntArrayList;
import it.unimi.dsi.fastutil.ints.IntArrays;
import it.unimi.dsi.fastutil.ints.IntOpenHashSet;
import it.unimi.dsi.fastutil.io.BinIO;
import it.unimi.dsi.fastutil.objects.ObjectArrayList;
import it.unimi.dsi.fastutil.objects.ObjectOpenHashSet;
import it.unimi.dsi.lang.ObjectParser;
import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.webgraph.GraphClassParser;
import it.unimi.dsi.webgraph.ImmutableGraph;
import it.unimi.dsi.webgraph.LazyIntIterators;
import it.unimi.dsi.webgraph.NodeIterator;
import it.unimi.dsi.webgraph.ImmutableGraph.LoadMethod;
import it.unimi.dsi.webgraph.labelling.ArcLabelledImmutableGraph;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Arrays;
import java.util.Random;

import org.apache.log4j.Logger;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.SimpleJSAP;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;

public class EstimateVotingCentrality {
	private final static Logger LOGGER = Util.getLogger( EstimateVotingCentrality.class );

	private static final boolean ASSERTS = true;
	private static final boolean DEBUG = false;
	
	private final int[][] succ;
	private final int[] choice;
	private final int[][] choiceT;
	private final double[] result;
	private final int n;
	private final Random random = new Random( System.currentTimeMillis() );
	public double alpha = .5;
	
	public EstimateVotingCentrality( final ArcLabelledImmutableGraph graph ) {
		throw new UnsupportedOperationException( "Labels are still unsupported" );
	}
	
	public EstimateVotingCentrality( final ImmutableGraph graph ) {
		final ObjectArrayList<int[]> succ = new ObjectArrayList<int[]>();
		int [] s;
		int n = 0;
		for( NodeIterator nodeIterator = graph.nodeIterator(); nodeIterator.hasNext(); ) {
			nodeIterator.nextInt();
			s = new int[ nodeIterator.outdegree() ];
			LazyIntIterators.unwrap( nodeIterator.successors(), s );
			succ.add( s );
			n++;
		}

		this.n = n;
		this.succ = succ.toArray( new int[ n ][] );
		choice = new int[ n ];
		choiceT = new int[ n ][];
		result = new double[ n ];
	}
	
	public void generateChoice() {
		final int n = this.n;
		final int[] choice = this.choice;
		final int[][] succ = this.succ;
		final int[] indegree = new int[ n ];

		// Isolated nodes vote for themselves
		for( int x = n; x-- != 0; ) indegree[ choice[ x ] = succ[ x ].length > 0 ? succ[ x ][ random.nextInt( succ[ x ].length ) ] : x ]++;
		for( int x = n; x-- != 0; ) choiceT[ x ] = new int[ indegree[ x ] ];
		if ( DEBUG ) System.err.println( "Indegree: " + Arrays.toString( indegree ) );
		IntArrays.fill( indegree, 0 );
			
		for( int x = n; x-- != 0; ) choiceT[ choice[ x ] ][ indegree[ choice[ x ] ]++ ] = x;
		
		if ( ASSERTS ) for( int i = 0; i < n; i++ ) for( int x: choiceT[ i ] ) choice[ x ] = i;
		
		if ( DEBUG ) {
			System.err.println( "Choice: " + Arrays.toString( choice ) );
			System.err.println( "ChoiceT: " + Arrays.deepToString( choiceT ) );
		}
	}

	public double[] computeRank() {
		final int n = this.n;
		final int[] choice = this.choice;
		final ObjectOpenHashSet<int[]> cycles = new ObjectOpenHashSet<int[]>();
		
		final double[] result = this.result;
		final byte[] marked = new byte[ n ]; // 0 = unknown, 1 = visiting, 2 = visited, and it's on a cycle, 3 = visited, and it's not on a cycle
		
		for( int i = 0; i < n; i++ ) {
			if ( marked[ i ] == 0 ) {
				int x = i;
				while( marked[ x ] == 0 ) {
					marked[ x ] = 1;
					x = choice[ x ];
				}

				if ( marked [ x ] == 1 ) {
					int y = x, l = 0;
					// Count cycle length
					do l++; while( ( y = choice[ y ] ) != x );

					final int[] cycle = new int[ l ];
					l = 0;
					// Add this cycle
					do {
						cycle[ l++ ] = y;
						marked[ y ] = 2; // Marked-cycle state
					} while( ( y = choice[ y ] ) != x );

					cycles.add( cycle );
				}

				x = i;
				while( marked[ x ] == 1 ) {
					marked[ x ] = 3;
					x = choice[ x ];
				}
			}
		}

		if ( ASSERTS ) {
			final IntOpenHashSet cycleNodes = new IntOpenHashSet();
			for( int[] cycle : cycles ) {
				
				for( int i = cycle.length; i-- != 0; ) {
					cycleNodes.addAll( IntArrayList.wrap( cycle ) );
					assert marked[ cycle[ i ] ] == 2;
					assert choice[ cycle[ i ] ] == cycle[ ( i + 1 ) % cycle.length ];
				}
			}

			for( int i = n; i-- != 0; ) assert cycleNodes.contains( i ) || marked[ i ] == 3 : i + " -> " + marked[ i ];
		}

		if ( DEBUG ) {
			System.err.println( "Choice: " + Arrays.toString( choice ) );
			System.err.println( "Mark: " + Arrays.toString( marked ) );
			System.err.print( "Cycles:" );
			for( int[] cycle: cycles ) System.err.print( " " + Arrays.toString( cycle ) );
			System.err.println();
		}
		
		for( int[] cycle: cycles ) {
			final double[] treeScore = new double[ cycle.length ];
			for( int i = cycle.length; i-- != 0; ) treeScore[ i ] = visit( cycle[ i ], cycle[ ( i - 1 + cycle.length ) % cycle.length ] );
			for( int i = cycle.length; i-- != 0; ) {
				double score = 0;
				for( int j = cycle.length; j-- != 0; ) score += treeScore[ ( i - j + cycle.length ) % cycle.length ] * Math.pow( alpha, j );
				score = score * ( 1 -alpha ) / ( n * ( 1 - Math.pow( alpha, cycle.length ) ) );
				result[ cycle[ i ] ] = score;
			}
		}

		if ( ASSERTS ) {
			double sum = 0;
			for( double r: result ) sum += r;
			assert sum - 1.0 <= 10E-6 : sum + " != " + 1;
		}
		
		return result;
	}

	public double[] computeVotingCentrality( final int maxIter ) {
		return computeVotingCentrality( maxIter, null );
	}
	
	public double[] computeVotingCentrality( final int maxIter, ProgressLogger pl ) {
		double[] curr = new double[ n ], c = new double[ n ];
		double t, y;
		if ( pl != null ) pl.start( "Computing centrality..." );
		pl.expectedUpdates = maxIter;
		
		for( int i = 0; i < maxIter; i++ ) {
			generateChoice();
			final double[] rank = computeRank();
			double delta = 0, deltaRel = 0;
			for( int j = rank.length; j-- != 0; ) {
				delta = Math.max( delta, Math.abs( curr[ j ] - i * rank[ j ] ) );
				deltaRel = Math.max( deltaRel, 1 + rank[ j ] / curr[ j ] );
				y = rank[ j ] - c[ j ];
				t = curr[ j ] + y;
				c[ j ] = ( t - curr[ j ] ) - y;
				curr[ j ] = t;
			}

			if ( pl != null ) {
				pl.update();
				if ( i != 0 ) pl.logger.info( delta / ( i * ( i + 1 ) )  + "; " + 100.0 * ( deltaRel * i / ( i + 1 ) - 1 ) + "%" );
			}
		}
		
		if ( pl != null ) pl.done();

		for( int i = curr.length; i-- != 0; ) curr[ i ] /= maxIter;
		return curr;
	}
	
	private double visit( int x, int noY ) {
		double result = 1;
		for( int i = choiceT[ x ].length; i-- != 0; ) if ( choiceT[ x ][ i ] != noY ) result += alpha * visit( choiceT[ x ][ i ] );
		this.result[ x ] = result * ( 1 - alpha) / n;
		return result;
	}
	
	private double visit( int x ) {
		double result = 1;
		for( int i = choiceT[ x ].length; i-- != 0; ) result += alpha * visit( choiceT[ x ][ i ] );
		this.result[ x ] = result * ( 1 - alpha) / n;
		return result;
	}
	
	public static void main( String args[] ) throws IllegalArgumentException, SecurityException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, IOException, JSAPException, ClassNotFoundException, InstantiationException {
		String scores;
		Class<?> graphClass;
	
		SimpleJSAP jsap = new SimpleJSAP( EstimateVotingCentrality.class.getName(), "Estimate centrality of a graph.",
				new Parameter[] {
						new FlaggedOption( "graphClass", GraphClassParser.getParser(), null, JSAP.NOT_REQUIRED, 'g', "graph-class", "Forces a Java class for the source graph." ),
						new FlaggedOption( "spec", new ObjectParser(), JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, 's', "spec", "A specification of the form <ImmutableGraphImplementation>(arg,arg,...)." ),
						new FlaggedOption( "maxIter", JSAP.INTEGER_PARSER, Integer.toString( 1000 ), JSAP.NOT_REQUIRED, 'i', "max-iter", "Maximum number of iterations."),
						new Switch( "once", '1', "once", "Use the read-once load method to read a graph from standard input." ),
						new Switch( "label", 'l', "label", "The graph is labelled, and the integer value associated to the well-known key must be used as a weight during the voting process." ),
						new FlaggedOption( "logInterval", JSAP.LONG_PARSER, Long.toString( ProgressLogger.DEFAULT_LOG_INTERVAL ), JSAP.NOT_REQUIRED, 'l', "log-interval", "The minimum time interval between activity logs in milliseconds." ),
						new UnflaggedOption( "basename", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.REQUIRED, JSAP.NOT_GREEDY, "The basename of the graph, unless --once is specified, in which case it is immaterial." ),
						new UnflaggedOption( "scores", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, JSAP.NOT_GREEDY, "The scores; if missing, scores will just be logged." ),
					}		
				);
		
		JSAPResult jsapResult = jsap.parse( args );
		if ( jsap.messagePrinted() ) return;

		graphClass = jsapResult.getClass( "graphClass" );
		final String basename = jsapResult.getString( "basename" );
		final String spec = jsapResult.getString( "spec" );
		final boolean once = jsapResult.getBoolean( "once" );
		final boolean label = jsapResult.getBoolean( "label" );
		final int maxIter = jsapResult.getInt( "maxIter" );
		scores = jsapResult.getString( "scores" ); 

		final ImmutableGraph graph;
		final ProgressLogger pl = new ProgressLogger( LOGGER, jsapResult.getLong( "logInterval" ) );

		if ( graphClass != null ) {
			if ( spec != null ) {
				System.err.println( "Options --graph-class and --spec are incompatible" );
				return;
			}
			graph = once ? (ImmutableGraph)graphClass.getMethod( LoadMethod.ONCE.toMethod(), InputStream.class ).invoke( null, System.in ) : 
				(ImmutableGraph)graphClass.getMethod( LoadMethod.OFFLINE.toMethod(), CharSequence.class ).invoke( null, basename );
		}
		else {
			if ( spec == null ) graph = once ? ImmutableGraph.loadOnce( System.in ) : ImmutableGraph.loadOffline( basename, pl );
			else graph = (ImmutableGraph)jsapResult.getObject( "spec" );
		}

		if ( label && ! ( graph instanceof ArcLabelledImmutableGraph ) ) throw new IllegalArgumentException( "The --label option requires a labelled graph." );
		
		final EstimateVotingCentrality estimateVotingCentrality = label ? new EstimateVotingCentrality( (ArcLabelledImmutableGraph)graph ) : new EstimateVotingCentrality( graph );
		final double[] rank = estimateVotingCentrality.computeVotingCentrality( maxIter, pl ); 
		
		if ( scores != null ) BinIO.storeDoubles( rank, scores );
		else for( double d: rank ) System.out.println( d );
	}

}