package voting;

/*		 
 * Voting in social networks
 *
 * Copyright (C) 2009-2010 Paolo Boldi, Francesco Bonchi, Carlos Castillo, Sebastiano Vigna 
 *
 *  This library is free software; you can redistribute it and/or modify it
 *  under the terms of the GNU Lesser General Public License as published by the Free
 *  Software Foundation; either version 2.1 of the License, or (at your option)
 *  any later version.
 *
 *  This library is distributed in the hope that it will be useful, but
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */


import it.unimi.dsi.Util;
import it.unimi.dsi.lang.ObjectParser;
import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.webgraph.GraphClassParser;
import it.unimi.dsi.webgraph.ImmutableGraph;
import it.unimi.dsi.webgraph.Transform;
import it.unimi.dsi.webgraph.ImmutableGraph.LoadMethod;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;

import org.apache.log4j.Logger;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.SimpleJSAP;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;

public class CountLeafChains {
	private final static Logger LOGGER = Util.getLogger( CountLeafChains.class );
	
	public static int computeLeafChains( final ImmutableGraph graph ) {
		final ImmutableGraph transpose = Transform.transpose( graph );
		final int n = graph.numNodes();
		
		int count = 0;
		for( int i = 0; i < n; i++ ) {
			if ( transpose.outdegree( i ) == 0 && graph.outdegree( i ) > 0 ) {
				// Leaf	
				if ( transpose.outdegree( graph.successors( i ).nextInt() ) == 1 ) count++;
			}
		}
		
		return count;
	
	}
	
	
	public static void main( String args[] ) throws IllegalArgumentException, SecurityException, IllegalAccessException, InvocationTargetException, NoSuchMethodException, IOException, JSAPException {
		Class<?> graphClass;
	
		SimpleJSAP jsap = new SimpleJSAP( CountLeafChains.class.getName(), "Counts the number of chains starting at a leaf.",
				new Parameter[] {
						new FlaggedOption( "graphClass", GraphClassParser.getParser(), null, JSAP.NOT_REQUIRED, 'g', "graph-class", "Forces a Java class for the source graph." ),
						new FlaggedOption( "spec", new ObjectParser(), JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, 's', "spec", "A specification of the form <ImmutableGraphImplementation>(arg,arg,...)." ),
						new Switch( "once", '1', "once", "Use the read-once load method to read a graph from standard input." ),
						new UnflaggedOption( "basename", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.REQUIRED, JSAP.NOT_GREEDY, "The basename of the graph, unless --once is specified, in which case it is immaterial." ),
					}		
				);
		
		JSAPResult jsapResult = jsap.parse( args );
		if ( jsap.messagePrinted() ) return;

		graphClass = jsapResult.getClass( "graphClass" );
		final String basename = jsapResult.getString( "basename" );
		final boolean once = jsapResult.getBoolean( "once" );
		final String spec = jsapResult.getString( "spec" );

		final ImmutableGraph graph;
		final ProgressLogger pl = new ProgressLogger( LOGGER );

		if ( graphClass != null ) {
			if ( spec != null ) {
				System.err.println( "Options --graph-class and --spec are incompatible" );
				return;
			}
			graph = once ? (ImmutableGraph)graphClass.getMethod( LoadMethod.ONCE.toMethod(), InputStream.class ).invoke( null, System.in ) : 
				(ImmutableGraph)graphClass.getMethod( LoadMethod.OFFLINE.toMethod(), CharSequence.class ).invoke( null, basename );
		}
		else {
			if ( spec == null ) graph = once ? ImmutableGraph.loadOnce( System.in ) : ImmutableGraph.loadOffline( basename, pl );
			else graph = (ImmutableGraph)jsapResult.getObject( "spec" );
		}
		System.out.println( computeLeafChains( graph ) );
	}

}