package es.yrbcn.graph.triangles;

import it.unimi.dsi.webgraph.ImmutableGraph;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.Arrays;

import cern.jet.random.engine.MersenneTwister;

/** This is an abstract algorithm for counting the number of triangles
 *  for the nodes in a graph.
 */

public abstract class TrianglesAlgorithm  {
	
	/** Default width */
	public final short DEFAULT_WIDTH = 32;

	/** The maximum distance for the iterations */
	public short maxDistance = 1;

	/** The graph. */
	public ImmutableGraph graph;

	/** The number of nodes. */
	public int numNodes = 0;

	/** done or not */
	public boolean done = false;

	/** Estimation */
	public double triangles[] = null;

	/** Random number generator; all random numbers should be created using this generator */
	public MersenneTwister random = null;

	public TrianglesAlgorithm() {
		throw new IllegalArgumentException( "Do not use" );
	}

	/** Creates a new TrianglesAlgorithm run; the algorithm
	 * will count how many triangles each element has
	 *
	 * @param g the graph.
	 * @param width the number of bits to use (32)
	 * @param seed the random seed for the random number generator
	 * @param maxDistance the distance to explore
	 */

	public TrianglesAlgorithm( final ImmutableGraph graph, int seed, short maxDistance ) {
		this.graph = graph;
		this.numNodes = graph.numNodes();
		this.random = new MersenneTwister( seed );
		this.maxDistance = maxDistance;

		// Initialize estimates to a negative value
		triangles = new double[numNodes];
		Arrays.fill( triangles, -1 );
	}

	public boolean done() {
		return done;
	}

	/** Initializes one run of the algorithm */
	protected abstract void init();

	/** Does one step of the algorithm; this is called until done() 
	 * @throws FileNotFoundException 
	 * @throws IOException */
	public abstract void step() throws FileNotFoundException, IOException;
	
	public abstract void countTriangles() throws IOException;
}
