package es.yrbcn.graph.triangles;
import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.webgraph.ImmutableGraph;
import it.unimi.dsi.webgraph.LazyIntIterator;
import it.unimi.dsi.webgraph.NodeIterator;

import java.util.Arrays;
import java.util.Hashtable;
import java.util.concurrent.TimeUnit;

import org.slf4j.LoggerFactory;




public class SamplingTrianglesAlgorithm extends TrianglesAlgorithm {

	public static final int DEFAULT_SAMPLE_SIZE = 100000;

	public int sampledNodeid[] = null;

	public int sampledTriangles[] = null;

	public Hashtable<Integer,Integer> sampledHash = null;

	public int sampleSize = DEFAULT_SAMPLE_SIZE;

	public int sampledDegree[] = null;

	public int sampledNeighbors[] = null;

	public int sampledNeighborsListStart[] = null;

	public int maxDegreeInGraph = 0;

	boolean sampleAtRandom = true;

	int sampleStartingPoint = 0;
	
	/**
	 * Creates a new TrianglesAlgorithm run; the algorithm is a functional
	 * ranking in which the operator used is OR
	 * 
	 * @param g
	 *            the graph.
	 * @param width
	 *            the number of bits to use (32)
	 * @param seed
	 *            the random seed for the random number generator
	 */

	public SamplingTrianglesAlgorithm(final ImmutableGraph graph,
			int seed, short maxDistance) {
		super(graph, seed, maxDistance);
		if (maxDistance > 1) {
			throw new IllegalArgumentException(
					"Not implemented at more than distance 1");
		}
		if (numNodes < sampleSize) {
			sampleSize = numNodes;
		}
		setRandomSampling();
	}

	public void setSampleSize(int ss) {
		sampleSize = ss;
		if (numNodes-sampleStartingPoint < sampleSize) {
			sampleSize = numNodes-sampleStartingPoint;
			System.err.println( "Sample size adjusted to " + sampleSize );
		}
	}

	public void setDeterministicSampling( int startingPoint ) {
		sampleStartingPoint = startingPoint;
		setSampleSize( sampleSize );
		sampleAtRandom = false;
		
	}

	public void setRandomSampling() {
		sampleStartingPoint = 0;
		sampleAtRandom = true;
	}

	public void init() {

		/* Create sample */
		sampledHash = new Hashtable<Integer,Integer>();
		sampledNodeid = new int[sampleSize];
		sampledDegree = new int[sampleSize];
		sampledTriangles = new int[sampleSize];
		Arrays.fill(sampledNodeid, -1);
		Arrays.fill(sampledDegree, 0);
		Arrays.fill(sampledTriangles, 0);
		
		if (sampleAtRandom) {
			for (int i = 0; i < sampleSize; i++) {
				int sampled_nodeid = (Math.abs(random.nextInt()) % numNodes);
				/* Sample without repetition */
				while (sampledHash.containsKey(new Integer(sampled_nodeid))) {
					sampled_nodeid = (Math.abs(random.nextInt()) % numNodes);
				}
				sampledNodeid[i] = sampled_nodeid;
				sampledHash.put(new Integer(sampledNodeid[i]), new Integer(i));
			}
		} else {
			for (int i = 0; i < sampleSize; i++ ) {
				int sampled_nodeid = sampleStartingPoint + i;
				sampledNodeid[i] = sampled_nodeid;
				sampledHash.put(new Integer(sampledNodeid[i]), new Integer(i));
			}
			
		}

		/* Get degree of nodes */
		System.err.println("Getting degree of nodes");
		NodeIterator nodes = graph.nodeIterator();
		int src;
		ProgressLogger pl = new ProgressLogger(LoggerFactory.getLogger("readgraph"),
				ProgressLogger.TEN_SECONDS, TimeUnit.MILLISECONDS, "nodes");
		pl.expectedUpdates = numNodes;
		pl.start();
		
		int total_sampled_degree = 0;
		
		while ( nodes.hasNext() ) {
			src = nodes.nextInt();
			// Tmp degree, including possible self-loops
			int aux_degree = nodes.outdegree();
			
			// Real degree
			int degree = 0;
			LazyIntIterator destL = nodes.successors();
			for (int i = 0; i < aux_degree; i++) {
				int dest = destL.nextInt();
				if(dest != src) {
					degree++;
				}
			}
			if (degree > maxDegreeInGraph) {
				maxDegreeInGraph = degree;
			}
			if (sampledHash.containsKey(new Integer(src))) {
				int sampleNum = sampledHash.get(new Integer(src))
						.intValue();
				sampledDegree[sampleNum] = degree;
				total_sampled_degree += degree;
			}
			pl.update();

		}
		pl.done();

		/* Get memory for adjacency lists */
		System.err
				.println("Getting memory for the total degree of sampled nodes ("
						+ total_sampled_degree + ")");
		sampledNeighborsListStart = new int[sampleSize];
		Arrays.fill(sampledNeighborsListStart, 0);
		sampledNeighbors = new int[total_sampled_degree];

		/* Load the adjacency lists */
		nodes = graph.nodeIterator();
		System.err
				.println("Loading in memory the adjacency list of the sampled nodes");
		pl = new ProgressLogger(LoggerFactory.getLogger("readgraph"),
				ProgressLogger.TEN_SECONDS, TimeUnit.MILLISECONDS, "nodes");
		pl.expectedUpdates = numNodes;
		pl.start();
		
		int next_ptr = 0;
		
		while ( nodes.hasNext() ) {
			src = nodes.nextInt();
			if (isSampled(src)) {
				int sampleNum = getSampleNumber(src);
				sampledNeighborsListStart[sampleNum] = next_ptr;
				int outdeg = nodes.outdegree();
				LazyIntIterator destL = nodes.successors();
				for (int i = 0; i < outdeg; i++) {
					int dest = destL.nextInt();
					if(dest != src) {
						sampledNeighbors[next_ptr++] = dest;
					}
				}
			}
			pl.update();
		}
		pl.done();

	}

	/**
	 * Calculation step; at each step I should have an estimation of the
	 * clustering coefficient for that particular step
	 */
	public void step() {
		
		System.err.println();
		NodeIterator nodes = graph.nodeIterator();
		ProgressLogger pl = new ProgressLogger(LoggerFactory.getLogger("readgraph"),
				ProgressLogger.TEN_SECONDS, TimeUnit.MILLISECONDS, "nodes");
		pl.expectedUpdates = numNodes;
		pl.start();
		
		LazyIntIterator destL;
		int dest_list[] = new int[maxDegreeInGraph];
		
		int src;
		while ( nodes.hasNext() ) {
			src = nodes.nextInt();
			destL = nodes.successors();
			int aux_degree = nodes.outdegree();
			int degree = 0;
			for (int i = 0; i < aux_degree; i++) {
				int dest = destL.nextInt();
				if( dest!=src ) {
					dest_list[degree++] = dest;
				}
			}
			
			for (int i = 0; i < degree; i++) {
				int dest = dest_list[i];
				if (isSampled(dest)) {
					int sampleNum = getSampleNumber(dest);
					int matches = getMatches(dest_list, degree, sampleNum);
					sampledTriangles[sampleNum] += matches;
				}
			}
			pl.update();
		}
		pl.done();
		
		countTriangles();
	}
	
	public void countTriangles() {
		/* Triangles where counted twice. I will divide by 2 and then store */
		for (int i = 0; i < sampleSize; i++) {
			sampledTriangles[i] /= 2;
			triangles[sampledNodeid[i]] = sampledTriangles[i];
		}
		done = true;
	}

	/** Shows sampled elements */
	public void dumpSample() {
		for (int i = 0; i < sampleSize; i++) {
			System.err.println("Sampled element: " + sampledNodeid[i]
					+ " Triangles: " + sampledTriangles[i]);
			System.err.println(" Adjacency list start "
					+ sampledNeighborsListStart[i] + " end "
					+ (sampledNeighborsListStart[i] + sampledDegree[i]));
			System.err.print(" Adjacency list: ");
			for (int j = sampledNeighborsListStart[i]; j < sampledNeighborsListStart[i]
					+ sampledDegree[i]; j++) {
				System.err.print(sampledNeighbors[j] + " ");
			}
			System.err.println();
		}
	}

	/**
	 * This routine assumes that the adjacency lists are sorted in ascendent
	 * order
	 */
	int getMatches(int list[], int len1, int sampleNum) {
		int pos1 = 0;
		int elem1 = list[pos1];
		if (sampledDegree[sampleNum] == 0) {
			return 0;
		}
	
		int pos2 = sampledNeighborsListStart[sampleNum];
		int elem2 = sampledNeighbors[pos2];
		int matches = 0;
		int end2 = sampledDegree[sampleNum] + pos2;
	
		while ((pos1 < len1) && (pos2 < end2)) {
			if (elem1 == elem2) {
				matches++;
				pos1++;
				if (pos1 < len1) {
					elem1 = list[pos1];
				}
				pos2++;
				if (pos2 < end2) {
					elem2 = sampledNeighbors[pos2];
				}
			} else if (elem1 < elem2) {
				pos1++;
				if (pos1 < len1) {
					elem1 = list[pos1];
				}
			} else {
				pos2++;
				if (pos2 < end2) {
					elem2 = sampledNeighbors[pos2];
				}
			}
		}
		return matches;
	}

	int getSampleNumber(int nodeid) {
		return sampledHash.get(new Integer(nodeid)).intValue();
	}

	boolean isSampled(int nodeid) {
		return sampledHash.containsKey(new Integer(nodeid));
	}
}
