package es.yrbcn.graph.triangles;

import it.unimi.dsi.logging.ProgressLogger;
import it.unimi.dsi.webgraph.ImmutableGraph;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.TimeUnit;

import org.apache.log4j.BasicConfigurator;
import org.slf4j.LoggerFactory;

import com.martiansoftware.jsap.FlaggedOption;
import com.martiansoftware.jsap.JSAP;
import com.martiansoftware.jsap.JSAPException;
import com.martiansoftware.jsap.JSAPResult;
import com.martiansoftware.jsap.Parameter;
import com.martiansoftware.jsap.SimpleJSAP;
import com.martiansoftware.jsap.Switch;
import com.martiansoftware.jsap.UnflaggedOption;

/**
 * Runs a triangle counting algorithm. The algorithm can be provided as an
 * argument in the commandline.
 * 
 * @author chato
 * 
 */
public class RunTriangles {

	/** Default distance */
	public static final short DEFAULT_MAX_DISTANCE = 1;

	/** The distance of the iterations */
	public static short maxDistance = DEFAULT_MAX_DISTANCE;

	/** Seed for the random number generator */
	public static final int DEFAULT_SEED = 131;

	/** The graph */
	public ImmutableGraph graph;

	/** The number of nodes. */
	public int numNodes;
	
	static boolean[] isKeyNode = null;
	public static int keyNodesCount = 0;
	
	public static final int PARTIAL_SAVE_EVERY = 10;

	@SuppressWarnings("unchecked")
	public static void main(final String arg[]) throws IOException, ClassNotFoundException, IllegalAccessException, NoSuchMethodException, InvocationTargetException,
			JSAPException {

		final SimpleJSAP jsap = new SimpleJSAP(
				RunTriangles.class.getName(),
				"Calculates an estimation of the number of triangles in a SYMMETRIC GRAPH.",
				new Parameter[] {
						new FlaggedOption("algorithm", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.REQUIRED, 'a', "algorithm",
								"Algorithm to use:\n" + 
								" bitbased    : Approximate algorithm based on bits, uses external memory\n" + 
								" mmbitbased  : Approximate algorithm based on bits, uses main memory only\n" + 
								" mmlbitbased : Approximate algorithm based on bits, uses main memory, one long per node (use for large graphs)\n" + 
								" sampling    : Exact algorithm, sample a set of nodes\n" + 
								" exact       : Exact algorithm for all nodes" ),
						new FlaggedOption("maxdistance", JSAP.SHORT_PARSER, Short.toString(DEFAULT_MAX_DISTANCE), JSAP.NOT_REQUIRED, 'm', "maxdistance",
								"Maximum distance to explore (NOT IMPLEMENTED)."),
						new Switch("file", 'f', "file", "Store all estimations on a file, with a name created from the base graph name."),
						new FlaggedOption("keynodes", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, 'k', "keynodes", "Store the obtained values only for these keys (node-ids)"),
						new Switch("partial", 't', "partial", "Write partial results every " + PARTIAL_SAVE_EVERY + " passes."),
						new Switch("offline", 'o', "offline", "Do not load graph in main memory (required for large graphs)."),
						new FlaggedOption("graph-class", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.NOT_REQUIRED, 'g', "graph-class", "Class to use for loading the graph"),
						new FlaggedOption("seed", JSAP.INTEGER_PARSER, Integer.toString(DEFAULT_SEED), JSAP.NOT_REQUIRED, 's', "seed",
								"Seed for the random number generator (NEVER use 0)"),
						new FlaggedOption("samplesize", JSAP.INTEGER_PARSER, Integer.toString(SamplingTrianglesAlgorithm.DEFAULT_SAMPLE_SIZE), JSAP.NOT_REQUIRED, 'z',
								"samplesize", "Sample size for '-a sampling'."),
						new FlaggedOption("samplebegin", JSAP.INTEGER_PARSER, Integer.toString(-1), JSAP.NOT_REQUIRED, 'b', "samplebegin",
								"First node to sample for '-a sampling'. If given, sample from this node 'samplesize' contiguous nodes. If not given, sample uniformly at random 'samplesize' nodes."),
						new FlaggedOption("passes", JSAP.INTEGER_PARSER, Integer.toString(BitbasedTrianglesAlgorithm.DEFAULT_MAX_PASSES), JSAP.NOT_REQUIRED, 'p',
								"passes", "Number of passes for '-a bitbased' and '-a mmbitbased'."),
						new UnflaggedOption("basename", JSAP.STRING_PARSER, JSAP.NO_DEFAULT, JSAP.REQUIRED, JSAP.NOT_GREEDY, "The base name for the graph to read") });

		// Parse arguments
		JSAPResult jsapResult = jsap.parse(arg);
		if (jsap.messagePrinted()) {
			return;
		}

		String algname = jsapResult.getString("algorithm");
		maxDistance = jsapResult.getShort("maxdistance");
		boolean save = jsapResult.getBoolean("file");
		String graphClassName = jsapResult.getString("graphclass");
		boolean loadOffline = jsapResult.getBoolean("offline");
		int seed = jsapResult.getInt("seed");
		int maxPasses = jsapResult.getInt("passes");
		int sampleSize = jsapResult.getInt("samplesize");
		int sampleBegin = jsapResult.getInt("samplebegin");
		boolean partialSave = jsapResult.getBoolean("partial");
		String basename = jsapResult.getString("basename");
		String keynodes = jsapResult.getString("keynodes");

		// Process graph class
		Class graphClass = null;
		if (graphClassName != null) {
			graphClass = Class.forName(graphClassName, true, ClassLoader.getSystemClassLoader());
		}

		// Load the graph file, cover all cases including
		// using the 'load' or 'loadOffline' method, and
		// using different graph classes
		System.err.print("Loading graph (please note that the graph must be symmetric). If the graph loading fails use -o or -Xmx1000M ... ");
		ImmutableGraph graph = null;
		BasicConfigurator.configure();
		if (graphClass != null) {
			graph = (ImmutableGraph) graphClass.getMethod(loadOffline ? "loadOffline" : "load", new Class[] { CharSequence.class }).invoke(graphClass,
					new Object[] { basename });
		} else {
			if (loadOffline) {
				graph = ImmutableGraph.loadOffline(basename);
			} else {
				graph = ImmutableGraph.load(basename);
			}
		}
		System.err.println("done.");

		// Size of the graph
		int numNodes = graph.numNodes();
		System.err.println("Number of nodes           : " + numNodes);

		if (maxDistance != DEFAULT_MAX_DISTANCE) {
			System.err.println("Max distance              : " + maxDistance);
		}
		if (!(algname.equals("exact") || (algname.equals("sampling") && sampleBegin != -1))) {
			System.err.println("Random number generator   : Mersenne Twister");
			System.err.println("Random seed               : " + seed);
		}

		if (algname.equals("sampling") && sampleBegin != -1) {
			int sampleEnd = sampleBegin + sampleSize - 1;
			System.err.println("Sampled nodes             : " + sampleBegin + " to " + sampleEnd);
		}
		
		// Key Nodes
		if( keynodes != null ) {
			readKeyNodes(keynodes, numNodes);
		}
		
		// More info
		if (save) {
			System.err.println("Will save number of triangles to a file");
		}
		if (partialSave) {
			System.err.println("Will write number of triangles every " + PARTIAL_SAVE_EVERY + " passes");
		}
		
		// Create algorithm
		TrianglesAlgorithm algorithm = null;

		if (algname.equals("exact")) {
			// Repeat many times the sampling algorithm with deterministic
			// sampling
			for (int startNode = 0; startNode < numNodes; startNode += sampleSize) {
				int endNode = startNode + sampleSize - 1;
				if (endNode >= numNodes) {
					endNode = numNodes;
				}
				System.err.println("Running from nodes " + startNode + " to " + endNode);
				algorithm = new SamplingTrianglesAlgorithm(graph, seed, maxDistance);
				((SamplingTrianglesAlgorithm) algorithm).setSampleSize(sampleSize);
				((SamplingTrianglesAlgorithm) algorithm).setDeterministicSampling(startNode);
				algorithm.init();
				while (!algorithm.done()) {
					algorithm.step();
				}
				if (save) {
					saveTriangles(basename, getDescription(algname, 0, seed, sampleSize, startNode) + "_NODES_" + startNode + "_" + endNode, algorithm );
				}
			}

		} else {

			if (algname.equals("sampling")) {
				algorithm = new SamplingTrianglesAlgorithm(graph, seed, maxDistance);
				((SamplingTrianglesAlgorithm) algorithm).setSampleSize(sampleSize);
				if (sampleBegin != -1) {
					((SamplingTrianglesAlgorithm) algorithm).setDeterministicSampling(sampleBegin);
				}

			} else if (algname.equals("bitbased") || algname.equals("mmbitbased") || algname.equals("mmlbitbased")) {

				if (algname.equals("bitbased")) {
					algorithm = new BitbasedTrianglesAlgorithm(graph, seed, maxDistance);
				} else if (algname.equals("mmbitbased")) {
					algorithm = new MainmemBitbasedTrianglesAlgorithm(graph, seed, maxDistance);
				} else if (algname.equals("mmlbitbased")) {
					algorithm = new MainmemlongBitbasedTrianglesAlgorithm(graph, seed, maxDistance);
				} else {
					System.err.println("Wrong algorithm");
					return;
				}
				((BitbasedTrianglesAlgorithm) algorithm).setMaxPasses(maxPasses);

			} else {
				System.err.println("Wrong algorithm");
				return;
			}

			// Initialize
			algorithm.init();

			// Run
			while (!algorithm.done()) {
				algorithm.step();

				if (partialSave && (!algorithm.done()) && (((BitbasedTrianglesAlgorithm) algorithm).currentPass % PARTIAL_SAVE_EVERY == 0)) {
					algorithm.countTriangles();
					
					saveTriangles(basename, getDescription(algname, ((BitbasedTrianglesAlgorithm) algorithm).currentPass, seed, 0, 0), algorithm );
				}
			}

			// Write number of triangles of each node
			if (save) {
				
				algorithm.countTriangles();
				saveTriangles(basename, getDescription(algname, maxPasses, seed, 0, 0), algorithm );
				
			}
		}

	}

	/**
	 * Reads key nodes
	 * 
	 * @param keynodes
	 * @param numNodes
	 * @throws FileNotFoundException
	 * @throws IOException
	 */
	static void readKeyNodes(String keynodes, int numNodes) throws FileNotFoundException, IOException {
		keyNodesCount = 0;
		// Load sampled nodes
		System.err.print("Loading key nodes ... ");
		isKeyNode = new boolean[numNodes];
		for (int i = 0; i < numNodes; i++) {
			isKeyNode[i] = false;
		}
		BufferedReader reader = null;
		reader = new BufferedReader(new FileReader(new File(keynodes)));
		String line = null;
		while ((line = reader.readLine()) != null) {
			isKeyNode[Integer.parseInt(line)] = true;
			keyNodesCount++;
		}
		reader.close();
		System.err.println(keyNodesCount + " key nodes read, only the estimation for them will be saved.");
	}

	static void saveTriangles(String basename, String descriptiveFilename, TrianglesAlgorithm algorithm ) throws IOException {

		String csvFile = basename + ".triangles_" + descriptiveFilename + ".csv";
		System.err.print("Writing estimations to " + csvFile + " :");
		ProgressLogger pl = new ProgressLogger(LoggerFactory.getLogger("write"), ProgressLogger.TEN_SECONDS, TimeUnit.MILLISECONDS, "nodes");
		pl.expectedUpdates = algorithm.numNodes; 
		pl.start();

		PrintWriter out = new PrintWriter(new BufferedWriter(new FileWriter(csvFile), 10 * 1024 * 1024)); // 10Mb
																											// Buffer
		for (int i = 0; i < algorithm.numNodes; i++) {
			
			if (algorithm.triangles[i] != -1 && ( isKeyNode == null || isKeyNode[i] == true ) ) { // Skip values without
				// estimation

				if (algorithm.triangles[i] < 0) {
					// This happened in MainmemBitbasedTrianglesAlgorithm when
					// the counter minima_matches
					// was an integer. It was changed to a double, but perhaps
					// this happens again. If you
					// get this warning, perhaps there is something else to fix
					// in MainmemBitbasedTrianglesAlgorithm
					System.err.println("Warning: there was an overflow or other type of error when computing " + "the triangles of node " + i
							+ ". This result was negative, omitting: " + algorithm.triangles[i]);
				} else {
					out.println(i + " " + algorithm.triangles[i]);
				}
			}
			pl.update();
		}
		out.close();
		pl.stop();
		System.err.println("done");
	}

	static String getDescription(String algname, int passes, int seed, int sampleSize, int sampleBegin) {
		// This description will be appended to all output filenames
		String description = "ALG_" + algname;
		
		if (!(algname.equals("sampling") || algname.equals("exact"))) {
			description += "_PASSES_" + passes;
		}

		if (algname.equals("sampling")) {
			description += "_SAMPLESIZE_" + sampleSize;
		}
		if (maxDistance != DEFAULT_MAX_DISTANCE) {
			description += "_DIST_" + maxDistance;
		}
		if (!(algname.equals("exact") || (algname.equals("sampling") && sampleBegin != -1))) {
			// If this is not deterministic
			description = description + "_MTSEED_" + seed;
		}

		if (algname.equals("sampling") && sampleBegin != -1) {
			int sampleEnd = sampleBegin + sampleSize - 1;
			description = description + "_NODES_" + sampleBegin + "_" + sampleEnd;
			System.err.println("Sampled nodes             : " + sampleBegin + " to " + sampleEnd);
		}

		return description;
	}

}
